package com.cashctrl.bookingimport;

import org.apache.commons.cli.*;

/**
 * Parse command line arguments with {@link org.apache.commons.cli.CommandLineParser}. <br>
 * The following arguments are possible: <br>
 * <b>-file</b> (stripe, paypal, sumup or payrexx csv file)<br>
 * <b>-debit</b> (debit account number 4 digits)<br>
 * <b>-credit</b> (credit account number 4 digits)<br>
 * @author Silian Barlogis
 */
public class CLI {

    /**
     * The parsed command line arguments.
     */
    private CommandLine cmd;

    /**
     * Parse command line arguments.
     * @param args arguments to parse
     * @return true if successful
     */
    public boolean parseArguments(String[] args) {
        if (args == null || args.length <= 0) {
            printHelp();
            return false;
        }

        try {
            CommandLineParser parser = new DefaultParser();
            cmd = parser.parse(getOptions(), args);
            if (cmd != null) {
                return true;
            }
        } catch (ParseException parseException) {
            System.out.print("Failed to parse the arguments! " + parseException);
        }
        return false;
    }

    /**
     * Check if option h or help is present.
     * @return true if either of those are present
     */
    public boolean wannaSeeHelp() {
        return cmd.hasOption("h") || cmd.hasOption("help");
    }

    /**
     * Print the help constructed out of the options using {@link HelpFormatter}.
     */
    public void printHelp() {
        new HelpFormatter().printHelp("CashCtrl - booking-import", getOptions());
    }

    /**
     * Validates whether important arguments are present.
     * @return true if all vital arguments are present
     */
    public boolean validateArguments() {
        return isArgumentValid("org") && isArgumentValid("key") && isArgumentValid("file")
                && isArgumentValid("revenue") && isArgumentValid("expense") && isArgumentValid("balance");
    }

    /**
     * Get the value of option.
     * @param option the option to get the value from
     * @return the value of the specified option
     */
    public String get(String option) {
        return cmd.getOptionValue(option);
    }

    /**
     * Validates if the argument is set and has a subsequent value.
     * @param arg argument to validate
     * @return true if argument is valid
     */
    private boolean isArgumentValid(String arg) {
        if (!cmd.hasOption(arg)) {
            System.out.print("Missing argument ");
            printArgInfo(arg);
            return false;
        }

        String value = cmd.getOptionValue(arg);
        if (value == null || value.isEmpty()) {
            System.out.print("Has no subsequent value ");
            printArgInfo(arg);
            return false;
        }

        return true;
    }

    /**
     * prints the argument info if it is a valid argument
     * @param arg the argument to print
     */
    private void printArgInfo(final String arg) {
        Option option = getOptions().getOption(arg);
        if (option != null) {
            System.out.println("-" + option.getOpt() + " <arg> " + option.getDescription());
        }
    }

    /**
     * Defines the command line options.
     * @return defined options
     */
    private Options getOptions() {
        Options options = new Options();
        options.addOption("h", "help", false, "Display help.");
        options.addOption("org", true, "Organisation name or URL.");
        options.addOption("key", true, "Organisation key.");
        options.addOption("file", true, "Stripe, PayPal, SumUp or Payrexx CSV file.");
        options.addOption("revenue", true, "Revenue account in CashCtrl.");
        options.addOption("expense", true, "Expense account in CashCtrl.");
        options.addOption("balance", true, "Balance account (Stripe, PayPal, SumUp or Payrexx) in CashCtrl.");
        options.addOption("invoiceField", true, "Invoice number referencing the transaction.");
        options.addOption("customerField", true, "Customer number referencing a person inside CashCtrl.");
        return options;
    }
}