package com.cashctrl.bookingimport.parser;

import com.cashctrl.bookingimport.Entry;
import org.apache.commons.csv.CSVParser;
import org.apache.commons.csv.CSVRecord;

import java.io.Closeable;
import java.util.ArrayList;
import java.util.List;

public class PayRexxParser extends BookingParser {
    private static final List<String> COLUMN_AMOUNT = List.of("Gross", "Brutto");
    private static final List<String> COLUMN_EMAIL = List.of("Email address", "E-Mail-Adresse");
    private static final List<String> COLUMN_FEE = List.of("Transaction fee", "Transaktionsgebühr");
    private static final List<String> COLUMN_CREATED = List.of("Date and time", "Datum und Zeit");
    private static final List<String> COLUMN_STATUS = List.of("Status");
    private static final List<String> COLUMN_CURRENCY = List.of("Currency", "Währung");
    private static final List<String> COLUMN_REFUNDED_AMOUNT = List.of("Refund", "Rückerstattung");

    private static final List<String> ACCEPTABLE_STATUS = List.of("Confirmed", "Partially refunded", "Refunded", "Bestätigt",
            "Teilrückerstattung", "Rückerstattet");

    /**
     * @param csvParser csv parser
     * @param input the input
     * @param revenueId revenue account id in CashCtrl
     * @param expenseId expense account id in CashCtrl
     * @param balanceId balance account id in CashCtrl
     */
    protected <T extends Closeable> PayRexxParser(CSVParser csvParser, T input, String revenueId, String expenseId, String balanceId) {
        super(csvParser, List.of(COLUMN_AMOUNT, COLUMN_EMAIL, COLUMN_FEE, COLUMN_CREATED, COLUMN_STATUS, COLUMN_CURRENCY, COLUMN_REFUNDED_AMOUNT),
                input, revenueId, expenseId, balanceId);
    }

    @Override
    public List<Entry> parse() {
        List<Entry> entries = new ArrayList<>();
        for (CSVRecord record : csvParser) {
            String status = getValue(record, COLUMN_STATUS);
            if (status == null || !isStatusValid(status)) {
                continue;
            }

            entries.add(new Entry(revenueId, expenseId, balanceId,
                    getValue(record, COLUMN_CREATED), getValue(record, COLUMN_EMAIL), getValue(record, COLUMN_AMOUNT),
                    getValue(record, COLUMN_FEE), getValue(record, COLUMN_CURRENCY), getValue(record, METADATA_COLUMN_CUSTOMER),
                    getValue(record, METADATA_COLUMN_INVOICE)));

            String refunded = getValue(record, COLUMN_REFUNDED_AMOUNT);
            if (refunded.equals("0")) {
                continue;
            }

            float refundedAmount = Float.parseFloat(refunded) * -1;
            entries.add(new Entry(revenueId, expenseId, balanceId,
                    getValue(record, COLUMN_CREATED), getValue(record, COLUMN_EMAIL), Float.toString(refundedAmount),
                    "0.0", getValue(record, COLUMN_CURRENCY), getValue(record, METADATA_COLUMN_CUSTOMER),
                    getValue(record, METADATA_COLUMN_INVOICE)));
        }
        return entries;
    }

    /**
     * Check if status is acceptable for parsing
     * @param status status
     * @return true if status is acceptable
     */
    protected boolean isStatusValid(String status) {
        for (String s : ACCEPTABLE_STATUS) {
            if (s.equalsIgnoreCase(status)) {
                return true;
            }
        }
        return false;
    }
}