package com.cashctrl.bookingimport.parser;

import com.cashctrl.bookingimport.Entry;
import com.cashctrl.bookingimport.Error;
import org.apache.commons.csv.CSVParser;
import org.apache.commons.csv.CSVRecord;

import java.io.Closeable;
import java.util.ArrayList;
import java.util.List;

/**
 * Parses a PayPal csv file.
 * @author Silian Barlogis
 */
public class PayPalParser extends BookingParser {
    private static final List<String> COLUMN_NAME = List.of("Name");
    private static final List<String> COLUMN_EMAIL = List.of("From Email Address", "Absender E-Mail-Adresse");

    private static final List<String> COLUMN_DATE = List.of("Date", "Datum");
    private static final List<String> COLUMN_TIME = List.of("Time", "Uhrzeit");
    private static final List<String> COLUMN_TIMEZONE = List.of("TimeZone", "Zeitzone");

    private static final List<String> COLUMN_AMOUNT = List.of("Gross", "Brutto");
    private static final List<String> COLUMN_FEE = List.of("Fee", "Entgelt");
    private static final List<String> COLUMN_STATUS = List.of("Status");
    private static final List<String> COLUMN_CURRENCY = List.of("Currency", "Währung");

    /**
     * Constructs a PayPal parser
     * @param <T> BufferedReader or InputStream
     * @param csvParser csv parser
     * @param input the input
     * @param revenueId revenue account id in CashCtrl
     * @param expenseId expense account id in CashCtrl
     * @param balanceId balance account id in CashCtrl
     */
    protected <T extends Closeable> PayPalParser(CSVParser csvParser, T input, String revenueId, String expenseId, String balanceId) {
        super(csvParser, List.of(COLUMN_NAME, COLUMN_EMAIL, COLUMN_DATE, COLUMN_TIME, COLUMN_TIMEZONE, COLUMN_AMOUNT, COLUMN_FEE,
                COLUMN_STATUS, COLUMN_CURRENCY), input, expenseId, revenueId, balanceId);
    }

    @Override
    public List<Entry> parse() {
        List<Entry> entries = new ArrayList<>();
        for (CSVRecord record : csvParser) {
            String status = getValue(record, COLUMN_STATUS);
            if (status == null || !status.equalsIgnoreCase("Completed") && !status.equalsIgnoreCase("Refunded")) {
                continue;
            }

            String timeZone = getValue(record, COLUMN_TIMEZONE);
            if (!timeZone.equals("CET") && !timeZone.equals("CEST")) {
                error.appendLine(new Error.Message("Warning: Timezone " + timeZone + " is not supported: " + getValue(record, COLUMN_NAME) + " " + getValue(record, COLUMN_EMAIL),
                        "Warnung: Zeitzone " + timeZone + " wird nicht unterstützt: " + getValue(record, COLUMN_NAME) + " " + getValue(record, COLUMN_EMAIL)));
            }

            Entry entry = new Entry(revenueId, expenseId, balanceId,
                    getValue(record, COLUMN_DATE), getValue(record, COLUMN_NAME) + " " + getValue(record, COLUMN_EMAIL), getValue(record, COLUMN_AMOUNT),
                    Float.toString(Math.abs(Float.parseFloat(getValue(record, COLUMN_FEE)))), getValue(record, COLUMN_CURRENCY), getValue(record, METADATA_COLUMN_CUSTOMER),
                    getValue(record, METADATA_COLUMN_INVOICE));

            entries.add(entry);
        }
        return entries;
    }
}