package com.cashctrl.bookingimport.web;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Enumeration;
import java.util.Locale;

/**
 * Web utilities.
 * @author Silian Barlogis
 */
public class WebUtils {
    private static final double MAX_ACCOUNT = Double.MAX_VALUE;
    private static final double MIN_ACCOUNT = 0.00;
    private static final int COOKIE_MAX_AGE = 60 * 60 * 24 * 365 * 10;

    /**
     * Get a cookie by name.
     * @param request the http request
     * @param name the cookie name
     * @return the cookie with the given name, returns an empty cookie if not found
     */
    public static Cookie getCookieByName(HttpServletRequest request, String name) {
        if (request != null && request.getCookies() != null) {
            for (Cookie cookie : request.getCookies()) {
                if (cookie.getName().equals(name)) {
                    return cookie;
                }
            }
        }
        return new Cookie(name, "");
    }

    /**
     * Get Cookie Value by name
     * @param request request
     * @param name cookie name
     * @return cookie value
     */
    public static String getCookieValue(HttpServletRequest request, String name) {
        return getCookieByName(request, name).getValue();
    }

    /**
     * Creates a new cookie
     * @param response response
     * @param name cookie name
     * @param value cookie value
     */
    public static void setCookie(HttpServletResponse response, String name, String value) {
        setCookies(response, new Cookie[]{new Cookie(name, value)});
    }

    /**
     * Add the cookies to the response and set their max age
     * @param response response
     * @param cookies cookies
     */
    public static void setCookies(HttpServletResponse response, Cookie[] cookies) {
        for (Cookie cookie : cookies) {
            cookie.setMaxAge(COOKIE_MAX_AGE);
            cookie.setPath("/");
            response.addCookie(cookie);
        }
    }

    /**
     * Validates if a account is valid.
     * @param account account to be validated
     * @return true if account is valid
     */
    public static boolean isAccountValid(double account) {
        return account < MAX_ACCOUNT && account >= MIN_ACCOUNT;
    }

    /**
     * Check if a string is null or empty
     * @param str string
     * @return true if string is null or empty
     */
    public static boolean isNullOrEmpty(String str) {
        return str == null || str.isEmpty();
    }

    /**
     * Get the preferred language, defaults to en if not possible
     * @param request request
     * @return preferred language
     */
    public static String getSupportedPreferredLanguage(HttpServletRequest request) {
        Enumeration<Locale> locales = request.getLocales();
        String language = "en";

        if (locales != null) {
            while (locales.hasMoreElements()) {
                Locale locale = locales.nextElement();
                String lang = locale.getLanguage();

                if (lang.equalsIgnoreCase("en")) {
                    language = lang;
                    break;
                } else if (lang.equalsIgnoreCase("de")) {
                    language = lang;
                    break;
                }
            }
        }

        return language;
    }
}