package com.cashctrl.bookingimport;

import com.cashctrl.bookingimport.parser.BookingParser;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;

import java.io.BufferedReader;
import java.io.Closeable;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.List;

/**
 * Starting-point of Booking import.
 * @author Silian Barlogis
 */
public final class BookingImport {

    private static final CLI cli = new CLI();
    private static final Error error = Error.getInstance();
    private static Organization organization;

    /**
     * Program main function
     * @param args commandline arguments
     */
    public static void main(String[] args) {
        if (!cli.parseArguments(args)) {
            return;
        }

        if (cli.wannaSeeHelp()) {
            cli.printHelp();
            return;
        }

        if (!cli.validateArguments()) {
            return;
        }

        error.setMode(Error.Mode.Full);
        error.setLanguage(Error.Language.English);

        System.out.println("------------ Start import ------------");
        for (String file : cli.get("file").split(" ")) {
            try {
                String org = cli.get("org");

                FileInputStream fileStream = new FileInputStream(file);
                InputStreamReader stream = new InputStreamReader(fileStream, StandardCharsets.UTF_8);
                BufferedReader bufferedReader = new BufferedReader(stream, fileStream.available());

                /*  Mark the current position (beginning of the file, since nothing has been read yet)
                    Set the size of the file stream as "read a head limit".
                    This way it is still possible to return to the marker if "read a head limit" is the whole file.
                 */
                bufferedReader.mark(fileStream.available());

                if (importFromCSV(org, cli.get("key"), cli.get("invoiceField"),
                        cli.get("customerField"), cli.get("revenue"), cli.get("expense"), cli.get("balance"), bufferedReader)) {
                    System.out.println("Successfully imported " + file + " into " + org);
                } else {
                    System.out.println("Failed to import " + file + " into " + org);
                }
            } catch (IOException ignored) {
                System.out.println("error: file not found : " + cli.get("file"));
                return;
            }
        }

        System.out.println(error.get());
        error.clear();
    }

    /**
     * Imports a csv file or input stream into CashCtrl
     * @param revenue the revenue account number
     * @param expense the expense account number
     * @param balance the balance account number
     * @param input file-reader or input stream
     * @param <T> Input-stream or file-reader
     * @return true if successfully imported payments into CashCtrl
     */
    public static <T extends Closeable> boolean importFromCSV(String orgName, String key,
                                                              String invoiceField, String customerField,
                                                              String revenue, String expense,
                                                              String balance, T input) {
        organization = new Organization(orgName, key);
        Journal journal = new Journal();
        StringBuilder revenueId = new StringBuilder(), expenseId = new StringBuilder(), balanceId = new StringBuilder();
        if (!getAccountIds(revenue, expense, balance, revenueId, expenseId, balanceId,
                journal.list(organization, "/api/v1/account/list.json"))) {
            return false;
        }

        var parser = BookingParser.evaluateParser(revenueId.toString(), expenseId.toString(), balanceId.toString(), input);
        if (parser == null) {
            error.appendLine(new Error.Message("Error: Not a valid Stripe, PayPal, SumUp or Payrexx csv.", "Fehler: Keine gültige Stripe-, PayPal-, SumUp- oder Payrexx-csv-Datei."));
            return false;
        }

        parser.setMetadataFields(invoiceField, customerField);
        List<Entry> entries = parser.parse();
        if (entries == null || entries.size() == 0) {
            error.appendLine(new Error.Message("Error: No \"Paid\" payments found.", "Fehler: Keine bezahlten (Paid) Zahlungen gefunden."));
            return false;
        }

        entries.forEach((entry -> {
            if (!journal.createEntry(entry, organization)) {
                error.appendLine(new Error.Message("Error: Failed to create " + entry + " inside " + organization.getUrl() + ".",
                        "Fehler: Das erstellen von " + entry + " in " + organization.getUrl() + " ist fehlgeschlagen."));
            }
        }));
        parser.close();

        return true;
    }

    /**
     * Try to find the debit and credit ids in organization
     * @param revenue revenue account number
     * @param expense expense account number
     * @param balance balance account number
     * @param balanceId out revenue id
     * @param revenueId out expense id
     * @param expenseId out balance id
     * @param accounts all accounts from organization
     * @return true if all account ids where found
     */
    private static boolean getAccountIds(String revenue, String expense, String balance,
                                         StringBuilder revenueId, StringBuilder expenseId,
                                         StringBuilder balanceId, JsonArray accounts) {
        if (accounts == null) {
            error.appendLine(new Error.Message("Error: Failed to get accounts from organization " + organization.getUrl() + ".",
                    "Fehler: Konten von der Organisation " + organization.getUrl() + " können nicht abgerufen werden."));
            return false;
        }

        for (int idx = 0; idx < accounts.size(); idx++) {
            final JsonObject account = accounts.get(idx).getAsJsonObject();
            if (Journal.has(account, "number")) {
                String number = account.get("number").getAsString();

                if (number.equals(revenue)) {
                    revenueId.append(account.get("id").getAsInt());
                }

                if (number.equals(expense)) {
                    expenseId.append(account.get("id").getAsInt());
                }

                if (number.equals(balance)) {
                    balanceId.append(account.get("id").getAsInt());
                }

                if (!revenueId.isEmpty() && !expenseId.isEmpty() && !balanceId.isEmpty()) {
                    return true;
                }
            }
        }

        if (revenueId.isEmpty()) {
            error.appendLine(new Error.Message("Error: Could not find " + revenue + " revenue account.",
                    "Fehler: Konnte das Ertragskonto " + revenue + " nicht finden."));
        }
        if (expenseId.isEmpty()) {
            error.appendLine(new Error.Message("Error: Could not find " + expense + " expense account.",
                    "Fehler: Konnte das Aufwandskonto " + expense + " nicht finden."));
        }
        if (balanceId.isEmpty()) {
            error.appendLine(new Error.Message("Error: Could not find " + balance + " balance account.",
                    "Fehler: Konnte das Bilanzkonto " + balance + " nicht finden."));
        }

        return false;
    }
}