package com.cashctrl.bookingimport.parser;

import com.cashctrl.bookingimport.Entry;
import com.cashctrl.bookingimport.Error;
import org.apache.commons.csv.CSVParser;
import org.apache.commons.csv.CSVRecord;

import java.io.Closeable;
import java.util.ArrayList;
import java.util.List;

/**
 * Parses a SumUp csv file.
 * @author Silian Barlogis
 */
public class SumUpParser extends BookingParser {
    private static final List<String> COLUMN_DESCRIPTION = List.of("Description", "Beschreibung");
    private static final List<String> COLUMN_DATE = List.of("Date", "Datum");
    private static final List<String> COLUMN_AMOUNT = List.of("Total amount", "Betrag inkl. MwSt.");
    private static final List<String> COLUMN_FEE = List.of("Fee", "Gebühr");
    private static final List<String> COLUMN_TYPE = List.of("Transaction Type", "Zahlungsart");
    private static final List<String> COLUMN_STATUS = List.of("Status");

    // private static final List<String> COLUMN_CURRENCY = List.of("Currency", "Währung");

    /**
     * Constructs a SumUp parser
     * @param csvParser csv parser
     * @param input the input
     * @param revenueId revenue account id in CashCtrl
     * @param expenseId expense account id in CashCtrl
     * @param balanceId balance account id in CashCtrl
     * @param <T> BufferedReader or InputStream
     */
    protected <T extends Closeable> SumUpParser(CSVParser csvParser, T input, String revenueId, String expenseId, String balanceId) {
        super(csvParser, List.of(COLUMN_DESCRIPTION, COLUMN_DATE, COLUMN_AMOUNT, COLUMN_FEE, COLUMN_TYPE, COLUMN_STATUS),
                input, expenseId, revenueId, balanceId);
    }

    @Override
    public List<Entry> parse() {
        List<Entry> entries = new ArrayList<>();
        for (CSVRecord record : csvParser) {

            boolean isRefund = isRefund(record);
            if (record == null || (!isRefund && !isValidTransaction(record))) {
                continue;
            }

            try {
                // The fee column is empty if it is a refund
                // (TLDR refund has no extra fee, you won't get the fee or parts of it back for the original transaction either)
                String fee = isRefund ? "0.0" : getValue(record, COLUMN_FEE);

                entries.add(new Entry(revenueId, expenseId, balanceId, getValue(record, COLUMN_DATE), getValue(record, COLUMN_DESCRIPTION),
                        getValue(record, COLUMN_AMOUNT), fee, null, getValue(record, METADATA_COLUMN_CUSTOMER),
                        getValue(record, METADATA_COLUMN_INVOICE)));

            } catch (Exception e) {
                error.appendLine(new Error.Message("Error: Failed to parse amount values" + (error.getMode().equals(Error.Mode.Full) ? ": " + e : "."),
                        "Fehler: Beim parsen der Zeit" + (error.getMode().equals(Error.Mode.Full) ? ": " + e : ".")));
            }
        }
        return entries;
    }

    /**
     * Check if record is a refund
     * @param record record
     * @return true if record is a refund
     */
    private boolean isRefund(CSVRecord record) {
        String type = getValue(record, COLUMN_TYPE);
        return type.equals("Refund") || type.equals("Rückerstattung");
    }

    /**
     * Check if record is a successful transaction
     * @param record record
     * @return true if a successful transaction
     */
    private boolean isValidTransaction(CSVRecord record) {
        String status = getValue(record, COLUMN_STATUS);
        return status.equals("Successful") || status.equals("Erfolgreich");
    }
}