package com.cashctrl.bookingimport.parser;

import com.cashctrl.bookingimport.Entry;
import com.cashctrl.bookingimport.Error;
import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVParser;
import org.apache.commons.csv.CSVRecord;

import java.io.BufferedReader;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;

/**
 * Booking parser baseclass
 * @author Silian Barlogis
 * @see StripeParser
 * @see PayPalParser
 * @see SumUpParser
 */
@SuppressWarnings("deprecation")
public abstract class BookingParser {
    protected static final CSVFormat defaultFormat = CSVFormat.DEFAULT.withFirstRecordAsHeader().withIgnoreHeaderCase();
    protected static final Error error = Error.getInstance();
    protected final String revenueId;
    protected final String expenseId;
    protected final String balanceId;
    protected final Closeable input;

    protected final List<String> header = new ArrayList<>();
    protected final CSVParser csvParser;
    private final List<List<String>> columns;

    protected List<String> METADATA_COLUMN_INVOICE = List.of("");
    protected List<String> METADATA_COLUMN_CUSTOMER = List.of("");

    /**
     * @param <T> BufferedReader or InputStream
     * @param csvParser csv parser
     * @param input the input
     * @param expenseId expense account id in CashCtrl
     * @param revenueId revenue account id in CashCtrl
     * @param balanceId balance account id in CashCtrl
     */
    protected <T extends Closeable> BookingParser(CSVParser csvParser, List<List<String>> columns, T input,
                                                  String revenueId, String expenseId, String balanceId) {
        this.csvParser = csvParser;
        this.columns = columns;
        this.input = input;
        this.revenueId = revenueId;
        this.expenseId = expenseId;
        this.balanceId = balanceId;
        header.addAll(csvParser.getHeaderNames());

        /*
        byte nullCharacter = 0;
        for (String head : csvParser.getHeaderNames()) {
            header.add(head.replace("\"", "").replace("\uFEFF", "").
                    replace(Byte.toString(nullCharacter), ""));
        }
        */
    }

    /**
     * parses the csv payments
     * @return return a list of each entry
     */
    public abstract List<Entry> parse();

    /**
     * Determines the parser for the input stream and returns an instance of it
     * @param revenueId revenue account id in CashCtrl
     * @param expenseId expense account id in CashCtrl
     * @param balanceId balance account id in CashCtrl
     * @param input the input
     * @param <T> BufferedReader or InputStream
     * @return returns the matching parser for the given input stream
     */
    public static <T extends Closeable> BookingParser evaluateParser(String revenueId, String expenseId, String balanceId, T input) {
        CSVParser csv = getParser(input, defaultFormat);
        if (csv == null) {
            return null;
        }

        StripeParser stripeParser = new StripeParser(csv, input, revenueId, expenseId, balanceId);
        if (stripeParser.isValid()) {
            return stripeParser;
        }

        PayPalParser paypalParser = new PayPalParser(csv, input, revenueId, expenseId, balanceId);
        if (paypalParser.isValid()) {
            return paypalParser;
        }

        SumUpParser sumupParser = new SumUpParser(csv, input, revenueId, expenseId, balanceId);
        if (sumupParser.isValid()) {
            return sumupParser;
        }

        reset(input);
        csv = getParser(input, defaultFormat.withDelimiter(';'));

        PayRexxParser payrexxParser = new PayRexxParser(csv, input, revenueId, expenseId, balanceId);
        if (payrexxParser.isValid()) {
            return payrexxParser;
        }

        error.appendLine(new Error.Message("Error: The stream layout could not be determined.",
                "Fehler: Das Stream-Layout konnte nicht ermittelt werden."));
        return null;
    }

    /**
     * Returns a csv parser
     * @param input input
     * @param format format
     * @param <T> BufferedReader or InputStream
     * @return csv parser
     */
    private static <T extends Closeable> CSVParser getParser(T input, CSVFormat format) {
        CSVParser csv = null;
        try {
            if (input instanceof BufferedReader) {
                csv = format.parse((BufferedReader) input);
            } else if (input instanceof InputStream) {
                csv = format.parse(new InputStreamReader((InputStream) input, StandardCharsets.UTF_8));
            } else {
                throw new IllegalArgumentException("Input must be of type import java.io.BufferedReader; or import java.io.InputStream;");
            }
        } catch (IOException | IllegalArgumentException e) {
            error.appendLine(new Error.Message("Error: Could not read stream" + (error.getMode().equals(Error.Mode.Full) ? ", message:" + e : "."),
                    "Fehler: Konnte die Datei nicht lesen" + (error.getMode().equals(Error.Mode.Full) ? ", Nachricht: " + e : ".")));
        }
        return csv;
    }

    /**
     * Resets the stream buffer position
     * @param input stream
     * @param <T> BufferedReader or InputStream
     */
    private static <T extends Closeable> void reset(T input) {
        try {
            if (input instanceof InputStream stream && stream.markSupported()) {
                stream.reset();
            } else if (input instanceof BufferedReader reader && reader.markSupported()) {
                reader.reset();  // reset to marked position
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Close all handles
     */
    public void close() {
        try {
            if (csvParser != null && !csvParser.isClosed()) {
                csvParser.close();
            }

            if (input != null) {
                input.close();
            }
        } catch (IOException e) {
            if (error.getMode().equals(Error.Mode.Full)) {
                error.appendLine(new Error.Message("Error: Failed to close file handle.",
                        "Fehler: Beim schliessen der Datei ist ein fehler aufgetreten."));
            }
        }
    }


    /**
     * Get record value as string
     * @param record the record to get the value from
     * @param keys the header to get
     * @return value
     */
    protected String getValue(CSVRecord record, List<String> keys) {
        for (String key : keys) {
            if (record.isMapped(key)) {
                return record.get(key);
            }
        }
        return null;
    }

    /**
     * Set the metadata fields.
     * @param invoiceField invoice field
     * @param customerField customer field
     */
    public void setMetadataFields(String invoiceField, String customerField) {
        if (invoiceField != null && !invoiceField.isEmpty()) {
            this.METADATA_COLUMN_INVOICE = List.of(invoiceField + " (metadata)", invoiceField);
        }

        if (customerField != null && !customerField.isEmpty()) {
            this.METADATA_COLUMN_CUSTOMER = List.of(customerField + " (metadata)", customerField);
        }
    }

    /**
     * Validate if file headers match columns
     * @return true if valid
     */
    protected boolean isValid() {
        return csvParser != null && columns != null && validate();
    }

    /**
     * Validates the parser if it has the expected headers (columns).
     * @return true if the input matches the parser
     */
    protected boolean validate() {
        if (header.size() < columns.size()) {
            return false;
        }

        for (List<String> column : columns) {
            boolean has = false;
            for (String col : column) {
                if (header.contains(col)) {
                    has = true;
                    break;
                }
            }

            if (!has) {
                return false;
            }
        }
        return true;
    }
}