package com.cashctrl.bookingimport.parser;

import com.cashctrl.bookingimport.Entry;
import com.cashctrl.bookingimport.Error;
import org.apache.commons.csv.CSVParser;
import org.apache.commons.csv.CSVRecord;

import java.io.Closeable;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.TimeZone;

/**
 * Parses a Stripe csv file.
 * @author Silian Barlogis
 */
public class StripeParser extends BookingParser {
    private static final List<String> COLUMN_AMOUNT = List.of("Converted Amount");
    private static final List<String> COLUMN_DESCRIPTION = List.of("Description");
    private static final List<String> COLUMN_FEE = List.of("Fee");
    private static final List<String> COLUMN_CREATED = List.of("Created (UTC)");
    private static final List<String> COLUMN_STATUS = List.of("Status");
    private static final List<String> COLUMN_CURRENCY = List.of("Converted Currency");
    private static final List<String> COLUMN_REFUNDED_AMOUNT = List.of("Converted Amount Refunded");

    /**
     * Constructs a Stripe parser
     * @param revenueId revenue account id in CashCtrl
     * @param expenseId expense account id in CashCtrl
     * @param balanceId balance account id in CashCtrl
     * @param input the input
     * @param <T> BufferedReader or InputStream
     */
    protected <T extends Closeable> StripeParser(CSVParser csvParser, T input, String revenueId, String expenseId, String balanceId) {
        super(csvParser, List.of(COLUMN_AMOUNT, COLUMN_DESCRIPTION, COLUMN_FEE, COLUMN_CREATED, COLUMN_STATUS, COLUMN_CURRENCY, COLUMN_REFUNDED_AMOUNT), input, expenseId, revenueId, balanceId);
    }

    @Override
    public List<Entry> parse() {
        List<Entry> entries = new ArrayList<>();
        for (CSVRecord record : csvParser) {
            String status = getValue(record, COLUMN_STATUS);
            if (status == null || !status.equalsIgnoreCase("Paid")) {
                continue;
            }

            SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm");
            dateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));

            String created = getValue(record, COLUMN_CREATED);

            try {
                Date date = dateFormat.parse(created);
                dateFormat.setTimeZone(TimeZone.getTimeZone("Europe/Zurich"));
                created = dateFormat.format(date);

            } catch (ParseException e) {
                error.appendLine(new Error.Message("Error: Failed to parse time: " + getValue(record, COLUMN_DESCRIPTION),
                        "Fehler: Beim parsen der Zeit: " + getValue(record, COLUMN_DESCRIPTION)));
                continue;
            }

            entries.add(new Entry(revenueId, expenseId, balanceId,
                    created, getValue(record, COLUMN_DESCRIPTION), getValue(record, COLUMN_AMOUNT),
                    getValue(record, COLUMN_FEE), getValue(record, COLUMN_CURRENCY), getValue(record, METADATA_COLUMN_CUSTOMER),
                    getValue(record, METADATA_COLUMN_INVOICE)));

            String refunded = getValue(record, COLUMN_REFUNDED_AMOUNT);
            if (refunded.equals("0.00")) {
                continue;
            }

            // value sign flips because the Journal createEntry function treats negative amounts as a refund
            float refundedAmount = Float.parseFloat(refunded) * -1;

            //The fee you get back from the refund has already been deducted from the fee of the original transaction. (fee = 0.0)
            // e.g Amount 290, fee 8.71: but 50 got refunded so the fee above is only 7.26
            entries.add(new Entry(revenueId, expenseId, balanceId,
                    created, getValue(record, COLUMN_DESCRIPTION), Float.toString(refundedAmount),
                    "0.0", getValue(record, COLUMN_CURRENCY), getValue(record, METADATA_COLUMN_CUSTOMER),
                    getValue(record, METADATA_COLUMN_INVOICE)));
        }
        return entries;
    }
}