package com.cashctrl.bookingimport.web;

import com.cashctrl.bookingimport.BookingImport;
import com.cashctrl.bookingimport.Error;
import javax.servlet.ServletException;
import javax.servlet.annotation.MultipartConfig;
import javax.servlet.annotation.WebServlet;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import javax.servlet.http.Part;

import java.io.IOException;
import java.util.Collection;
import java.util.EnumSet;
import java.util.concurrent.atomic.AtomicReference;

/**
 * Import Servlet handles the request and responses of the CashCtrl web import
 * @author Silian Barlogis
 */
@WebServlet(name = "ImportServlet", urlPatterns = {"/en/import", "/de/import"})
@MultipartConfig(maxFileSize = 1024 * 1024 * 25,
        maxRequestSize = 1024 * 1024 * 40, fileSizeThreshold = 1024 * 1024)
public class ImportServlet extends HttpServlet {

    private static final Error error = Error.getInstance();
    private static final String direct = "bookingimport.jsp";
    private String revenue;
    private String expense;
    private String balance;
    private String organisationName;
    private String organisationKey;
    private String invoiceField;
    private String customerField;

    private enum State {
        SUCCESS,
        WARNING,
        ERROR
    }

    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        final HttpSession session = request.getSession();
        if (session == null) {
            response.sendRedirect(direct);
            return;
        }

        revenue = request.getParameter("revenue");
        expense = request.getParameter("expense");
        balance = request.getParameter("balance");
        organisationName = request.getParameter("organisationName");
        organisationKey = request.getParameter("organisationKey");
        invoiceField = request.getParameter("invoiceField");
        customerField = request.getParameter("customerField");

        if (validate(session)) {
            response.sendRedirect(direct);
            return;
        }

        setCookies(response);

        final Collection<Part> parts = getMultipart(request, session);
        if (parts == null) {
            response.sendRedirect(direct);
            return;
        }

        setLanguage(request);
        parts.forEach(part -> importCSV(session, part));
        response.sendRedirect(direct);
    }

    /**
     * Import the csv into cashctrl
     * @param session session
     * @param part multipart/filestream
     */
    private void importCSV(HttpSession session, Part part) {
        if (part.getName().equals("csvFile")) {
            try {
                if (BookingImport.importFromCSV(organisationName, organisationKey, invoiceField, customerField, revenue,
                        expense, balance, part.getInputStream())) {
                    if (error.get().length() > 0) {
                        // warning
                        session.setAttribute("alert-type", "warning");
                        session.setAttribute("alert-head", getHead(State.WARNING));
                        session.setAttribute("alert-message", prepareMessage(error.get()));
                    } else {
                        // success
                        session.setAttribute("alert-type", "success");
                        session.setAttribute("alert-head", getHead(State.SUCCESS));
                        session.setAttribute("alert-message", "");
                    }

                } else {
                    // error
                    session.setAttribute("alert-type", "danger");
                    session.setAttribute("alert-head", getHead(State.ERROR));
                    session.setAttribute("alert-message", prepareMessage(error.get()));
                }

                error.clear();
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }

    /**
     * Set error message language
     * @param request http servlet request
     */
    private void setLanguage(HttpServletRequest request) {
        String lang = WebUtils.getCookieValue(request, "lang");
        final AtomicReference<Error.Language> errorLanguage = new AtomicReference<>(Error.Language.English);
        EnumSet.allOf(Error.Language.class).forEach(l -> {
            if (l.abbreviation().equalsIgnoreCase(lang)) {
                errorLanguage.set(l);
            }
        });

        error.setLanguage(errorLanguage.get());
        error.setMode(Error.Mode.Brief);
    }

    private Collection<Part> getMultipart(HttpServletRequest request, HttpSession session) throws ServletException {
        final Collection<Part> parts;
        try {
            parts = request.getParts();
            if (parts == null || parts.size() < 1) {
                throw new IOException();
            }
        } catch (IOException e) {
            session.setAttribute("alert-type", "danger");
            session.setAttribute("alert-head", getHead(State.ERROR));
            session.setAttribute("alert-message", "Failed to get csv file from request, please try again!");
            return null;
        }
        return parts;
    }

    /**
     * Set cookies
     * @param response http servlet response
     */
    private void setCookies(HttpServletResponse response) {
        // set cookies
        final Cookie[] cookies = new Cookie[]{
                new Cookie("revenue", revenue), new Cookie("expense", expense),
                new Cookie("balance", balance), new Cookie("organisationName", organisationName),
                new Cookie("invoiceField", invoiceField), new Cookie("customerField", customerField)
        };

        WebUtils.setCookies(response, cookies);
    }

    /**
     * Validate input arguments.
     * @param session http session
     * @return true if arguments are valid
     */
    private boolean validate(HttpSession session) {
        try {
            if (!WebUtils.isAccountValid(Double.parseDouble(revenue)) || !WebUtils.isAccountValid(Double.parseDouble(expense)) ||
                    !WebUtils.isAccountValid(Double.parseDouble(balance)) || WebUtils.isNullOrEmpty(organisationName) ||
                    WebUtils.isNullOrEmpty(organisationKey)) {
                throw new IllegalArgumentException();
            }
        } catch (Exception e) {
            session.setAttribute("alert-type", "danger");
            session.setAttribute("alert-head", getHead(State.ERROR));
            session.setAttribute("alert-message", "Failed to validate input parameter, please try again!");
            return true;
        }
        return false;
    }

    /**
     * Prepare the alert message
     * @param msg the message
     * @return prepared message
     */
    private static String prepareMessage(String msg) {
        Error.Language language = error.getLanguage();
        if (language.equals(Error.Language.German)) {
            msg = msg.replace("\n", "<br>")
                    .replace("Fehler:", "<strong>Fehler: </strong>")
                    .replace("Feld:", "<strong>Feld: </strong>")
                    .replace("Nachricht:", "<strong>Nachricht: </strong>")
                    .replace("Status:", "<strong>Status: </strong>");

        } else if (language.equals(Error.Language.English)) {
            msg = msg.replace("\n", "<br>")
                    .replace("Error:", "<strong>Error: </strong>")
                    .replace("Field:", "<strong>Field. </strong>")
                    .replace("Message:", "<strong>Message: </strong>")
                    .replace("Status:", "<strong>Status: </strong>");
        }

        return msg;
    }

    /**
     * Get head message
     * @param state state
     * @return head message
     */
    private static String getHead(State state) {
        Error.Language language = error.getLanguage();

        return switch (state) {
            case SUCCESS ->
                    language.equals(Error.Language.English) ? "Successfully imported into CashCtrl!" : "Erfolgreich in CashCtrl importiert!";
            case WARNING ->
                    language.equals(Error.Language.English) ? "Warnings occurred during import!" : "Beim importieren sind Warnungen aufgetreten!";
            case ERROR ->
                    language.equals(Error.Language.English) ? "An error occurred during import!" : "Beim Importieren ist ein Fehler aufgetreten!";
        };
    }
}