package com.cashctrl.bookingimport;

/**
 * Singleton for storing errors
 * @author Silian Barlogis
 */
public final class Error {
    /**
     * Represents the languages
     */
    public enum Language {
        English(1, "en"),
        German(2, "de");

        @SuppressWarnings("unused")
        final int id;
        final String abbreviation;

        Language(int id, String abbreviation) {
            this.id = id;
            this.abbreviation = abbreviation;
        }

        @SuppressWarnings("unused")
        public int id() {
            return id;
        }

        public String abbreviation() {
            return abbreviation;
        }
    }

    /**
     * Represents the error modes
     * Brief - Only important stuff used for website
     * Full - The full error message with exceptions
     */
    public enum Mode {
        Brief,
        Full
    }

    /**
     * Represents an error message with different languages
     * @param english message in english
     * @param german message in german
     */
    public record Message(String english, String german) {
    }

    private final StringBuilder error = new StringBuilder();
    private static Error instance = null;
    private Language language = Language.English;
    private Mode mode = Mode.Full;

    /**
     * Private constructor.
     */
    private Error() {
    }

    /**
     * Get error instance, create one if this is null.
     * @return instance of this
     */
    public static Error getInstance() {
        if (instance == null) {
            instance = new Error();
        }
        return instance;
    }

    /**
     * Append an error with a new line
     * @param error the error to append
     */
    public void appendLine(Message error) {
        append(switchLanguage(error) + '\n');
    }

    /**
     * Append a error
     * @param error the error to append
     */
    private void append(String error) {
        this.error.append(error);
    }

    /**
     * Get all errors
     * @return all errors
     */
    public String get() {
        return error.toString();
    }

    /**
     * Clear all errors
     */
    public void clear() {
        error.setLength(0);
    }

    /**
     * Switches the message to the current language
     * @param msg the message
     * @return the message in the correct language
     */
    private String switchLanguage(Message msg) {
        switch (language) {
            case English -> {
                return msg.english();
            }
            case German -> {
                return msg.german();
            }
        }
        throw new IllegalArgumentException();
    }

    public Language getLanguage() {
        return language;
    }

    public void setLanguage(final Language language) {
        this.language = language;
    }

    public Mode getMode() {
        return mode;
    }

    public void setMode(final Mode mode) {
        this.mode = mode;
    }
}