package com.cashctrl.bookingimport;

import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;

import java.net.http.HttpResponse;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

public class Journal {
    /**
     * HttpMethods instance for POST and GET requests.
     */
    private static final HttpMethods http = HttpMethods.getInstance();
    private static final Error error = Error.getInstance();
    private static final Gson gson = new Gson();
    private static final List<String> supportedCurrencies = new ArrayList<>(
            Arrays.asList("aud", "btc", "cad", "chf", "cny", "dkk", "eur",
                    "gbp", "inr", "jpy", "rub", "sek", "thb", "usd"));
    private static final int limit = 99999;
    private JsonArray people = null;
    private JsonArray currencies = null;

    /**
     * Create new entry in organization.
     * @param entry the entry to create
     * @param organization the organization to create the entry in
     * @return true if create entry was successful
     */
    public boolean createEntry(Entry entry, Organization organization) {
        HashMap<String, String> data = new HashMap<>();

        // flip accounts if amount is signed (refund)
        boolean refund = false;
        float amount = Float.parseFloat(entry.amount());
        if (amount < 0.0f) {
            refund = true;
            data.put("creditId", entry.balanceId());
            data.put("debitId", entry.revenueId());
        } else {
            data.put("creditId", entry.revenueId());
            data.put("debitId", entry.balanceId());
        }

        data.put("amount", Float.toString(Math.abs(amount)));

        String date = entry.date();
        if (date != null && !date.isEmpty()) {
            String[] dateSplit = date.split(" ");
            if (dateSplit.length > 0) {
                data.put("dateAdded", dateSplit[0]);
            }
        }

        if (entry.description() != null && !entry.description().isEmpty()) {
            data.put("title", entry.description());
        }

        if (entry.referenceNr() != null && !entry.referenceNr().isEmpty()) {
            data.put("reference", entry.referenceNr());
        }

        if (entry.customerNr() != null && !entry.customerNr().isEmpty()) {
            if (people == null) {
                people = list(organization, "/api/v1/person/list.json");
            }

            if (people != null && people.size() > 0) {
                for (int idx = 0; idx < people.size(); idx++) {
                    JsonObject person = people.get(idx).getAsJsonObject();
                    if (has(person, "nr") && has(person, "id") && person.get("nr").getAsString().equals(entry.customerNr())) {
                        data.put("associateId", Integer.toString(person.get("id").getAsInt()));
                    }
                }
            }
        }

        if (entry.currency() != null && !entry.currency().isEmpty()) {
            if (currencies == null) {
                currencies = list(organization, "/api/v1/currency/list.json");
            }

            if (currencies != null && currencies.size() > 0) {
                boolean exists = false;
                for (int idx = 0; idx < currencies.size(); idx++) {
                    JsonObject currency = currencies.get(idx).getAsJsonObject();
                    if (has(currency, "code") && has(currency, "id") && has(currency, "rate") && currency.get("code").getAsString().equalsIgnoreCase(entry.currency())) {
                        data.put("currencyId", Integer.toString(currency.get("id").getAsInt()));

                        if (!supportedCurrencies.contains(currency.get("code").getAsString().toLowerCase())) {
                            data.put("currencyRate", Float.toString(currency.get("rate").getAsFloat()));
                        }

                        exists = true;
                        break;
                    }
                }

                if (!exists) {
                    HttpResponse<String> response = createCurrency(entry.currency(), organization);
                    if (checkResponse(response)) {
                        currencies = null;

                        int currencyId = insertId(response);
                        if (currencyId != -1) {
                            data.put("currencyId", Integer.toString(currencyId));
                            if (!supportedCurrencies.contains(entry.currency().toLowerCase())) {
                                data.put("currencyRate", "1");
                            }
                        } else {
                            error.appendLine(new Error.Message("Error: Failed to get the insert id of the created currency",
                                    "Fehler: Konnte die Einfüge-ID der erstellten Währung nicht abrufen"));
                        }
                    } else {
                        error.appendLine(new Error.Message("Error: Failed to create the currency",
                                "Fehler: Fehler beim erstellen der Währung"));
                    }
                }
            }
        }

        HttpResponse<String> response = http.post(organization, "/api/v1/journal/create.json", data);
        if (!checkResponse(response)) {
            return false;
        }

        Float fee = Float.parseFloat(entry.fee());
        if (fee.equals(0.0f)) {
            return true;
        }

        data.put("creditId", refund ? entry.expenseId() : entry.balanceId());
        data.put("debitId", refund ? entry.balanceId() : entry.expenseId());
        data.put("amount", Float.toString(Math.abs(fee)));

        response = http.post(organization, "/api/v1/journal/create.json", data);
        return checkResponse(response);
    }

    /**
     * Create a new currency with the given code.
     * @param code currency code like "CHF"
     * @param organization the organization where the object should get created in
     * @return response data as json array
     */
    private HttpResponse<String> createCurrency(String code, Organization organization) {
        HashMap<String, String> data = new HashMap<>();
        data.put("code", code);
        if (!supportedCurrencies.contains(code.toLowerCase())) {
            data.put("rate", "1");
        }

        return http.post(organization, "/api/v1/currency/create.json", data);
    }

    /**
     * List entries in the given organization.
     * @param organization the organization to list from
     * @return response data as json array
     */
    public JsonArray list(Organization organization, String endpoint) {
        HashMap<String, String> filter = new HashMap<>();
        filter.put("limit", Integer.toString(limit));
        HttpResponse<String> response = http.get(organization, endpoint, filter);
        if (response != null && response.statusCode() == 200) {
            JsonObject body = gson.fromJson(response.body(), JsonObject.class);
            if (has(body, "data")) {
                return body.get("data").getAsJsonArray();
            }
        }
        return null;
    }

    /**
     * check response if any errors occurred on the update or creation process.
     * @param response http response from a post request
     * @return return true if successful
     */
    protected boolean checkResponse(HttpResponse<String> response) {
        if (response == null || response.statusCode() != 200)
            return false;

        JsonObject jsonResponse = gson.fromJson(response.body(), JsonObject.class);
        if (has(jsonResponse, "success")) {
            // update was successfully
            if (jsonResponse.get("success").getAsBoolean())
                return true;

            // check for validation errors
            if (has(jsonResponse, "errors")) {
                JsonArray errors = jsonResponse.get("errors").getAsJsonArray();
                for (int idx = 0; idx < errors.size(); idx++) {
                    JsonObject errorObj = errors.get(idx).getAsJsonObject();
                    if (error.getMode().equals(Error.Mode.Full)) {
                        String field = (has(errorObj, "field") ? errorObj.get("field").getAsString() : "null");
                        error.appendLine(new Error.Message("Field: " + field, "Feld: " + field));
                    }
                    String message = (has(errorObj, "message") ? errorObj.get("message").getAsString() : "null");
                    error.appendLine(new Error.Message("Message: " + message, "Nachricht: " + message));
                }
            }

            //check for other messages
            if (has(jsonResponse, "message")) {
                String message = jsonResponse.get("message").getAsString();
                error.appendLine(new Error.Message("Message: " + message, "Nachricht: " + message));
            }
        }

        return false;
    }

    /**
     * Check if a json object is not null, has the property and is not json-null.
     * @param object the object that will be validated
     * @param property the has property
     * @return true if object is valid and has property which is not json null
     */
    public static boolean has(JsonObject object, String property) {
        return object != null && object.has(property) && !object.get(property).isJsonNull();
    }

    /**
     * Get the insert id of a created object.
     * @param response the response with body
     * @return the insert id, return -1 if failed to get insert id
     */
    protected int insertId(HttpResponse<String> response) {
        if (response == null || response.statusCode() != 200)
            return -1;

        JsonObject jsonResponse = gson.fromJson(response.body(), JsonObject.class);
        return has(jsonResponse, "insertId") ? jsonResponse.get("insertId").getAsInt() : -1;
    }
}