package com.cashctrl.bookingimport;

import java.io.IOException;
import java.net.ProxySelector;
import java.net.URI;
import java.net.URLEncoder;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.HashMap;

/**
 * Singleton for sending Http Post and Get requests.
 * @author Silian Barlogis
 */
public final class HttpMethods {

    /**
     * Constants for how many request can be sent before it throttles down to <b>{@value THROTTLE} requests in {@value WAIT} seconds</b>.
     */
    public static final int BURST_FIRE = 100;
    public static final int THROTTLE = 10;
    public static final int WAIT = 1000;

    private static final Error error = Error.getInstance();

    /**
     * Httpclient for sending requests and retrieve their response, configured to always redirect, except from https to http.
     */
    private final HttpClient client = HttpClient.newBuilder()
            .version(HttpClient.Version.HTTP_2)
            .followRedirects(HttpClient.Redirect.NORMAL)
            .proxy(ProxySelector.getDefault())
            .build();

    /**
     * Singleton instance.
     */
    private static HttpMethods instance;

    /**
     * The amount of requests.
     */
    private int requestCounter = 0;

    /**
     * Private constructor.
     */
    private HttpMethods() {
    }

    /**
     * Get http methods instance, create one if this is null.
     * @return instance of this
     */
    public static HttpMethods getInstance() {
        if (instance == null) {
            instance = new HttpMethods();
        }
        return instance;
    }

    /**
     * Get request with query parameters, encodes the parameters.
     * @param org organization to get from
     * @param uri the endpoint to ask
     * @param params the query parameter as {@link HashMap}
     * @return the response with header and body or null
     */
    public HttpResponse<String> get(Organization org, String uri, HashMap<String, String> params) {
        StringBuilder parameter = new StringBuilder();
        if (params == null) {
            return null;
        }

        params.put("lang", error.getLanguage().abbreviation);
        parameter.append("?");
        params.forEach((key, value) -> {
            parameter.append(URLEncoder.encode(key, StandardCharsets.UTF_8));
            parameter.append("=");
            parameter.append(URLEncoder.encode(value, StandardCharsets.UTF_8));
            if (params.size() > 1) {
                parameter.append("&");
            }
        });

        return get(org, uri + parameter);
    }

    /**
     * Get request.
     * The Organization must contain a valid authorization and domain.
     * @param org organization to get from
     * @param uri the endpoint to ask
     * @return the response with header and body or null
     */
    private HttpResponse<String> get(Organization org, String uri) {
        HttpResponse<String> response = null;
        maybeDelayRequest();

        try {
            HttpRequest request = HttpRequest.newBuilder()
                    .uri(URI.create(org.getUrl() + uri))
                    .timeout(Duration.ofSeconds(20))
                    .header("Content-Type", "application/json")
                    .header("Authorization", org.getAuthorization())
                    .GET()
                    .build();

            requestCounter++;
            response = client.send(request, HttpResponse.BodyHandlers.ofString());
        } catch (IllegalStateException | IOException | InterruptedException exception) {
            String e = error.getMode().equals(Error.Mode.Full) ? exception.toString() : "";
            error.appendLine(new Error.Message("Error: Organisation not found. " + e, "Fehler: Organisation nicht gefunden. " + e));
        } finally {
            if (response == null || response.statusCode() != 200) {
                if (error.getMode().equals(Error.Mode.Full)) {
                    error.appendLine(new Error.Message("GET request failed!", "GET Anfrage fehlgeschlagen!"));
                }

                if (response != null) {
                    printResponse(response);
                }
            }
        }
        return response;
    }

    /**
     * Post request.
     * The Organization must contain a valid authorization and domain.
     * @param org organization to post to
     * @param uri endpoint to ask
     * @param data the payload to send
     * @return the response with header and body or null
     */
    public HttpResponse<String> post(Organization org, String uri, HashMap<String, String> data) {
        data.put("lang", error.getLanguage().abbreviation);
        HttpResponse<String> response = null;
        maybeDelayRequest();

        try {
            HttpRequest request = HttpRequest.newBuilder()
                    .uri(URI.create(org.getUrl() + uri))
                    .timeout(Duration.ofSeconds(20))
                    .header("Content-Type", "application/x-www-form-urlencoded")
                    .header("Authorization", org.getAuthorization())
                    .POST(ofData(data))
                    .build();

            requestCounter++;
            response = client.send(request, HttpResponse.BodyHandlers.ofString());
        } catch (IllegalStateException | IOException | InterruptedException exception) {
            String e = error.getMode().equals(Error.Mode.Full) ? exception.toString() : "";
            error.appendLine(new Error.Message("Error: Organisation not found. " + e, "Fehler: Organisation nicht gefunden. " + e));
        } finally {
            if (response == null || response.statusCode() != 200) {
                if (error.getMode().equals(Error.Mode.Full)) {
                    error.appendLine(new Error.Message("POST request failed!", "POST Anfrage fehlgeschlagen!"));
                }

                if (response != null) {
                    printResponse(response);
                }
            }
        }

        return response;
    }

    /**
     * Prepares and encodes the payload for the post request
     * @param data data payload
     * @return the payload prepared and encoded ready for post request
     */
    public HttpRequest.BodyPublisher ofData(HashMap<String, String> data) {
        StringBuilder dataStr = new StringBuilder();

        data.forEach((key, value) -> {
            if (dataStr.length() > 0) {
                dataStr.append("&");
            }

            dataStr.append(URLEncoder.encode(key, StandardCharsets.UTF_8));
            dataStr.append("=");
            dataStr.append(URLEncoder.encode(value, StandardCharsets.UTF_8));
        });

        return HttpRequest.BodyPublishers.ofString(dataStr.toString());
    }

    /**
     * Prints the status code and body
     * @param response the response that should be printed
     */
    private void printResponse(HttpResponse<String> response) {
        error.appendLine(new Error.Message("Status: " + response.statusCode(), "Status: " + response.statusCode()));
        error.appendLine(new Error.Message("Message: " + response.body(), "Nachricht: " + response.body()));
    }

    /**
     * Check if the next requests needs to be delayed
     */
    private void maybeDelayRequest() {
        if (requestCounter >= BURST_FIRE) {
            try {
                Thread.sleep(WAIT);
                requestCounter -= THROTTLE;
            } catch (InterruptedException interruptedException) {
                throw new RuntimeException(interruptedException);
            }
        }
    }
}