package com.cashctrl.orgsync;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

import java.net.http.HttpResponse;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

/**
 * Attachment holds functions for synchronizing attachments, it also serves as a group for main elements like
 * {@link com.cashctrl.orgsync.Person} or {@link com.cashctrl.orgsync.Article}
 * @author Silian Barlogis
 * @see com.cashctrl.orgsync.Organization
 * @see com.cashctrl.orgsync.File
 */
public abstract class Attachment extends SyncObject {

    protected final File file;
    protected final String updateAttachmentsEndpoint;

    /**
     * SyncObject constructor
     * @param alphaOrganization the alpha organization
     * @param betaOrganization the beta organization
     * @param listEndpoint endpoint for list action
     * @param readEndpoint endpoint for read action
     * @param createEndpoint endpoint for create action
     * @param updateEndpoint endpoint for update action
     * @param updateAttachmentsEndpoint endpoint for updating attachments
     * @param equalProperty name of the property for equality test
     * @param dependencyProperty name of the dependency property
     */
    protected Attachment(Organization alphaOrganization, Organization betaOrganization, String listEndpoint, String readEndpoint,
                         String createEndpoint, String updateEndpoint, String updateAttachmentsEndpoint, String equalProperty, String dependencyProperty) {
        super(alphaOrganization, betaOrganization, listEndpoint, readEndpoint, createEndpoint, updateEndpoint, equalProperty, dependencyProperty);
        this.updateAttachmentsEndpoint = updateAttachmentsEndpoint;
        this.file = File.first(alphaOrganization, betaOrganization);
    }

    /**
     * Attaches files to a record.
     * @param recordId the id of the record where the files are attached
     * @param fileIds the ids of the files to attach
     * @param organization the organization where the record is
     * @return true if successful
     */
    protected boolean attach(int recordId, List<Integer> fileIds, Organization organization) {
        HashMap<String, String> data = new HashMap<>();
        data.put("id", Integer.toString(recordId));

        StringBuilder filesStr = new StringBuilder();
        fileIds.forEach(fileId -> filesStr.append(fileId).append(','));
        data.put("fileIds", filesStr.toString());

        HttpResponse<String> response = http.post(organization, updateAttachmentsEndpoint, data, HttpResponse.BodyHandlers.ofString());
        return checkResponse(response, null, "update attachment");
    }

    /**
     * Updates the target with the attachments from source
     * @param syncData the synchronization data with source and target
     * @param targetOrganization the target organization
     */
    protected void updateAttachments(SyncData syncData, Organization targetOrganization) {
        JsonObject srcDetails = syncData.sourceDetails();
        JsonObject target = syncData.target();
        if (!has(srcDetails, "attachments") && target.has("id")) {
            return;
        }

        JsonArray attachments = srcDetails.get("attachments").getAsJsonArray();
        if (attachments == null || attachments.isEmpty()) {
            return;
        }

        List<Integer> files = new ArrayList<>();
        for (JsonElement a : attachments) {
            JsonObject attachment = (JsonObject) a;

            if (!has(attachment, "fileId") || !has(attachment, "name")) {
                continue;
            }

            JsonObject equalFile = file.equivalent(attachment.get("name").getAsString(), file.list(targetOrganization));
            if (!has(equalFile, "id") || !has(equalFile, "name")) {
                System.out.println("error: failed to update attachment, could not find equivalent file inside target organization: " + targetOrganization.name() +
                        " file name: " + attachment.get("name").getAsString());
                continue;
            }

            files.add(equalFile.get("id").getAsInt());
        }

        if (!attach(target.get("id").getAsInt(), files, targetOrganization)) {
            System.out.println("error: failed to attach " + files + " to " + target + " in " + targetOrganization);
        }
    }
}