package com.cashctrl.orgsync;

import java.io.IOException;
import java.net.ProxySelector;
import java.net.URI;
import java.net.URLEncoder;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.HashMap;

/**
 * Singleton for sending Http Post and Get requests.
 * @author Silian Barlogis
 */
public final class HttpMethods {
    /**
     * Constants for how many request can be sent before it throttles down to <b>{@value THROTTLE} requests in {@value WAIT} seconds</b>.
     */
    private static final int BURST_FIRE = 100;
    private static final int THROTTLE = 10;
    private static final int WAIT = 1000;
    /**
     * Singleton instance.
     */
    private static HttpMethods instance;
    /**
     * Httpclient for sending requests and retrieve their response, configured to always redirect, except from https to http.
     */
    private final HttpClient client = HttpClient.newBuilder().version(HttpClient.Version.HTTP_2)
            .followRedirects(HttpClient.Redirect.NORMAL).proxy(ProxySelector.getDefault()).build();
    /**
     * The amount of requests.
     */
    private int requestCounter = 0;
    /**
     * Execution duration for the request, throws an exception if the request is not finished.
     */
    private Duration requestDuration = Duration.ofSeconds(20);

    /**
     * Private constructor.
     */
    private HttpMethods() {
    }

    /**
     * Get http methods instance, create one if this is null.
     * @return instance of this
     */
    public static HttpMethods instance() {
        if (instance == null) {
            instance = new HttpMethods();
        }
        return instance;
    }

    /**
     * Get request.
     * The Organization must contain a valid authorization and domain.
     * @param organization organization to get from
     * @param uri the endpoint to ask
     * @param parameter request parameter
     * @param handle request body handler
     * @param <T> http response type
     * @return the response with header and body or null
     */
    public <T> HttpResponse<T> get(Organization organization, String uri, HashMap<String, String> parameter,
                                   HttpResponse.BodyHandler<T> handle) {

        final StringBuilder encodedParameter = new StringBuilder();
        if (parameter != null && !parameter.isEmpty()) {
            encodedParameter.append("?");
            parameter.forEach((key, value) -> {
                encodedParameter.append(URLEncoder.encode(key, StandardCharsets.UTF_8)).append("=")
                        .append(URLEncoder.encode(value, StandardCharsets.UTF_8));
                if (parameter.size() > 1) {
                    encodedParameter.append("&");
                }
            });
        }

        URI url = URI.create(organization.url() + uri + encodedParameter);
        HttpRequest request = HttpRequest.newBuilder().uri(url).timeout(requestDuration)
                .header("Authorization", organization.authorization()).GET().build();

        maybeDelayRequest();
        requestCounter++;

        HttpResponse<T> response = null;
        try {
            response = client.send(request, handle);
        } catch (IllegalStateException | IOException | InterruptedException exception) {
            System.out.print("error: exception: " + exception);
        } finally {
            if (response == null || response.statusCode() != 200) {
                System.out.println("GET request failed!");
                if (response != null) {
                    printResponse(response);
                }
            }
        }
        return response;
    }

    /**
     * Post request.
     * The Organization must contain a valid authorization and domain.
     * @param organization organization to post to
     * @param uri endpoint to ask
     * @param data the payload to send
     * @param handle request body handler
     * @param <T> http response type
     * @return the response with header and body or null
     */
    public <T> HttpResponse<T> post(Organization organization, String uri, HashMap<String, String> data, HttpResponse.BodyHandler<T> handle) {

        final StringBuilder encodedData = new StringBuilder();
        if (data != null) {
            data.forEach((key, value) -> {
                if (!encodedData.isEmpty()) {
                    encodedData.append("&");
                }
                encodedData.append(URLEncoder.encode(key, StandardCharsets.UTF_8)).append("=")
                        .append(URLEncoder.encode(value, StandardCharsets.UTF_8));
            });
        }

        HttpRequest request = HttpRequest.newBuilder().uri(URI.create(organization.url() + uri))
                .timeout(requestDuration).header("Content-Type", "application/x-www-form-urlencoded")
                .header("Authorization", organization.authorization())
                .POST(HttpRequest.BodyPublishers.ofString(encodedData.toString())).build();

        maybeDelayRequest();
        requestCounter++;

        HttpResponse<T> response = null;
        try {
            response = client.send(request, handle);
        } catch (IllegalStateException | IOException | InterruptedException exception) {
            System.out.print("error: exception: " + exception);
        } finally {
            if (response == null || response.statusCode() != 200) {
                System.out.println("POST request failed!");
                if (response != null) {
                    printResponse(response);
                }
            }
        }

        return response;
    }

    /**
     * Put request
     * @param endpoint the endpoint url for the put request
     * @param mime the mime content type, leave blank if byte blob
     * @param data the data to send
     * @return the response with header and body or null
     */
    public <T> HttpResponse<T> put(String endpoint, String mime, byte[] data, HttpResponse.BodyHandler<T> handle) {
        HttpRequest request = HttpRequest.newBuilder()
                .uri(URI.create(endpoint)).timeout(Duration.ofSeconds(requestCounter))
                .header("Content-Type", mime.isEmpty() ? "application/octet-stream" : mime)
                .PUT(HttpRequest.BodyPublishers.ofByteArray(data)).build();

        maybeDelayRequest();
        requestCounter++;

        HttpResponse<T> response = null;
        try {
            response = client.send(request, handle);
        } catch (IllegalStateException | IOException | InterruptedException exception) {
            System.out.print("error: exception: " + exception);
        } finally {
            if (response == null || response.statusCode() != 200) {
                System.out.println("PUT request failed!");
                if (response != null) {
                    printResponse(response);
                }
            }
        }

        return response;
    }

    /**
     * Prints the status code and body
     * @param response the response that should be printed
     */
    private <T> void printResponse(HttpResponse<T> response) {
        System.out.println("Status: " + response.statusCode());
        System.out.println("Body: " + response.body());
    }

    /**
     * Check if the next requests needs to be delayed
     */
    private void maybeDelayRequest() {
        if (requestCounter >= BURST_FIRE - 5) {
            try {
                Thread.sleep(WAIT);
                requestCounter -= THROTTLE;
            } catch (InterruptedException interruptedException) {
                throw new RuntimeException(interruptedException);
            }
        }
    }

    @SuppressWarnings("unused")
    public Duration getRequestDuration() {
        return requestDuration;
    }

    public void setRequestDuration(final Duration requestDuration) {
        this.requestDuration = requestDuration;
    }
}