package com.cashctrl.orgsync;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;

import java.util.HashMap;
import java.util.List;

/**
 * File category API synchronize, list, read, create and update file categories.
 * @author Silian Barlogis
 * @see SyncObject
 * @see com.cashctrl.orgsync.Category
 */
public class FileCategory extends Category {

    /**
     * Category constructor, initializes all needed endpoints for category synchronization.
     * @param alphaOrganization alpha organization
     * @param betaOrganization beta organization
     */
    public FileCategory(Organization alphaOrganization, Organization betaOrganization) {
        super(alphaOrganization, betaOrganization, "/api/v1/file/category/list.json", "/api/v1/file/category/read.json",
                "/api/v1/file/category/create.json", "/api/v1/file/category/update.json", "/api/v1/file/category/tree.json");
    }

    @Override
    public State synchronize() {
        JsonArray alphaTree = tree(-1, alphaOrganization);
        JsonArray betaTree = tree(-1, betaOrganization);

        if (alphaTree == null || betaTree == null) {
            return State.Error;
        } else if (alphaTree.isEmpty() && betaTree.isEmpty()) {
            return State.NoChange;
        }

        int rootId = 0;
        JsonObject alphaRoot = alphaTree.get(rootId).getAsJsonObject();
        JsonObject betaRoot = betaTree.get(rootId).getAsJsonObject();
        if (alphaRoot == null || betaRoot == null) {
            return State.Error;
        } else if (!has(alphaRoot, "data") && !has(betaRoot, "data")) {
            return State.NoChange;
        }

        return super.synchronize(alphaRoot.get("data").getAsJsonArray(), betaRoot.get("data").getAsJsonArray(), rootId, rootId);
    }

    @SuppressWarnings("DuplicatedCode")
    @Override
    protected HashMap<String, String> parametrize(SyncData syncData) {
        JsonObject fileCategory = syncData.source();

        var fields = List.of(
                new Parameter("name", fileCategory, DataType.STRING),
                new Parameter("parentId", fileCategory, DataType.INT)
        );

        return parametrizeParams(fields);
    }
}