package com.cashctrl.orgsync;

import java.time.Duration;
import java.util.Base64;
import java.util.Date;
import java.util.HashMap;

/**
 * Represents an organization
 * @author Silian Barlogis
 */
public class Organization {

    private final Settings settings = new Settings();

    /**
     * Organization domain <b><a href="https://myOrg.cashctrl.com">https://myOrg.cashctrl.com</a></b>
     */
    private final String url;

    /**
     * Organization basic authorization with API kay
     */
    private final String authorization;

    /**
     * Organization raw name
     */
    private final String name;

    /**
     * Holds the datetime of the last synchronized delta
     */
    private long lastSynchronizedDelta = Duration.ofSeconds(10).toMillis();

    /**
     * Holds the datetime when this organization was last synchronized
     */
    private long lastSynchronized = 0;

    /**
     * Organization constructor
     * @param organizationName organization name
     * @param key organization key
     */
    @SuppressWarnings("HttpUrlsUsage")
    public Organization(String organizationName, String key) {
        name = organizationName;
        url = organizationName.contains("https://") || organizationName.contains("http://") ? organizationName : "https://" + organizationName + ".cashctrl.com";
        authorization = "Basic " + Base64.getEncoder().encodeToString((key + ":").getBytes());

        try {
            lastSynchronized = Long.parseLong(settings.value("LAST_SYNCHRONIZED", this));
        } catch (Exception exception) {
            System.out.println("error: failed to get LAST_SYNCHRONIZED setting from " + name);
        }
    }

    /**
     * Updates the LAST_SYNCHRONIZED setting
     */
    public void updateLastSynchronized() {
        try {
            HashMap<String, String> data = new HashMap<>();
            data.put("LAST_SYNCHRONIZED", Long.toString(new Date().getTime()));
            if (!settings.update(data, this)) {
                throw new Exception();
            }
        } catch (Exception exception) {
            System.out.println("error: failed to set LAST_SYNCHRONIZED setting inside organization " + name);
        }
    }

    public String url() {
        return url;
    }

    public String authorization() {
        return authorization;
    }

    public String name() {
        return name;
    }

    public long lastSynchronized() {
        return lastSynchronized + lastSynchronizedDelta;
    }

    public void lastSynchronizedDelta(long lastSynchronizedDelta) {
        this.lastSynchronizedDelta = lastSynchronizedDelta;
    }
}