package com.cashctrl.orgsync;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;

import java.net.http.HttpResponse;
import java.util.HashMap;

/**
 * Category API synchronize, list, read, create and update categories.
 * @author Silian Barlogis
 * @see SyncObject
 */
public abstract class Category extends SyncObject {

    /**
     * Endpoint for category tree.
     */
    protected final String treeEndpoint;

    /**
     * Category constructor, initializes all needed endpoints for category synchronization.
     * @param alphaOrganization alpha organization
     * @param betaOrganization beta organization
     * @param treeEndpoint endpoint for category tree
     */
    protected Category(Organization alphaOrganization, Organization betaOrganization, String listEndpoint, String readEndpoint,
                       String createEndpoint, String updateEndpoint, String treeEndpoint) {
        super(alphaOrganization, betaOrganization, listEndpoint, readEndpoint, createEndpoint, updateEndpoint, "name", "categoryId");
        this.treeEndpoint = treeEndpoint;
    }

    @Override
    public State synchronize() {
        int alphaParentId = -1;
        int betaParentId = -1;

        if (filter != null && filter.containsKey("category")) {
            JsonObject alphaCategory = equivalent(filter.get("category"), list(alphaOrganization));
            JsonObject betaCategory = equivalent(filter.get("category"), list(betaOrganization));

            if (isValid(alphaCategory) && isValid(betaCategory)) {
                alphaParentId = alphaCategory.get("id").getAsInt();
                betaParentId = betaCategory.get("id").getAsInt();
            } else {
                System.out.println("error: the category " + filter.get("category") + " does not exist in both organizations");
                return State.Error;
            }
        }

        JsonArray alphaTree = tree(alphaParentId, alphaOrganization);
        JsonArray betaTree = tree(betaParentId, betaOrganization);

        if (alphaTree == null || betaTree == null)
            return State.Error;
        else if (alphaTree.isEmpty() && betaTree.isEmpty())
            return State.NoChange;

        return synchronize(alphaTree, betaTree, alphaParentId, betaParentId);
    }

    /**
     * Recursive function to synchronize categories using tree synchronization.
     * @param alphaCategories array with alpha categories to synchronize (tree not list)
     * @param betaCategories array with beta categories to synchronize (tree not list)
     * @param alphaParentId the parent id of the alpha array categories
     * @param betaParentId the parent id of the beta array categories
     * @return success, error or no change
     */
    @SuppressWarnings("DuplicatedCode")
    protected State synchronize(JsonArray alphaCategories, JsonArray betaCategories,
                                int alphaParentId, int betaParentId) {
        for (int alphaIdx = 0; alphaIdx < alphaCategories.size(); alphaIdx++) {
            JsonObject alphaCategory = alphaCategories.get(alphaIdx).getAsJsonObject();
            if (!isValid(alphaCategory))
                continue;

            boolean existsInBeta = false;
            for (int betaIdx = 0; betaIdx < betaCategories.size(); betaIdx++) {
                JsonObject betaCategory = betaCategories.get(betaIdx).getAsJsonObject();
                if (!isValid(betaCategory))
                    continue;

                if (equals(alphaCategory, betaCategory)) {
                    existsInBeta = true;

                    long alphaTime = lastUpdated(alphaCategory);
                    long betaTime = lastUpdated(betaCategory);

                    if (alphaTime != betaTime) {
                        if (alphaTime > betaTime && alphaTime > alphaOrganization.lastSynchronized()) {
                            updateCategory(betaParentId, alphaCategory, betaCategory, betaOrganization);
                        } else if (betaTime > alphaTime && betaTime > betaOrganization.lastSynchronized()) {
                            updateCategory(alphaParentId, betaCategory, alphaCategory, alphaOrganization);
                        }
                    }
                }
            }

            if (!existsInBeta) {
                createCategory(betaParentId, alphaCategory, betaOrganization);
            }
        }

        for (int betaIdx = 0; betaIdx < betaCategories.size(); betaIdx++) {
            JsonObject categoryBeta = betaCategories.get(betaIdx).getAsJsonObject();
            if (!isValid(categoryBeta))
                continue;

            boolean existsInAlpha = false;
            for (int alphaIdx = 0; alphaIdx < alphaCategories.size(); alphaIdx++) {
                JsonObject categoryAlpha = alphaCategories.get(alphaIdx).getAsJsonObject();
                if (!isValid(categoryAlpha))
                    continue;

                if (equals(categoryAlpha, categoryBeta))
                    existsInAlpha = true;
            }

            if (!existsInAlpha) {
                createCategory(alphaParentId, categoryBeta, alphaOrganization);
            }
        }

        // sync child categories
        JsonArray newAlphaCategories = tree(alphaParentId, alphaOrganization);
        JsonArray newBetaCategories = tree(betaParentId, betaOrganization);

        if (newAlphaCategories.size() != newBetaCategories.size()) {
            System.out.println("error: failed to sync categories unknown cause");
            return State.Error;
        }

        for (int alphaIdx = 0; alphaIdx < newAlphaCategories.size(); alphaIdx++) {
            JsonObject alphaCategory = newAlphaCategories.get(alphaIdx).getAsJsonObject();
            if (!isValid(alphaCategory) || !has(alphaCategory, "leaf"))
                continue;

            JsonObject betaCategory = equivalent(alphaCategory, newBetaCategories);
            if (!isValid(betaCategory) || !has(betaCategory, "leaf"))
                continue;

            // has child categories ?
            if (isLeaf(alphaCategory) && isLeaf(betaCategory))
                continue;

            JsonArray alphaTree = has(alphaCategory, "data") ?
                    alphaCategory.get("data").getAsJsonArray() : new JsonArray();
            JsonArray betaTree = has(betaCategory, "data") ?
                    betaCategory.get("data").getAsJsonArray() : new JsonArray();

            synchronize(alphaTree, betaTree, alphaCategory.get("id").getAsInt(),
                    betaCategory.get("id").getAsInt());
        }
        return State.Success;
    }

    /**
     * Creates a new category inside target Organization with given parent id.
     * @param targetParentId parent id inside target organization
     * @param sourceCategory the date to create
     * @param targetOrganization the organization to create in
     */
    private void createCategory(int targetParentId, JsonObject sourceCategory, Organization targetOrganization) {
        if (has(sourceCategory, "parentId")) {
            sourceCategory.addProperty("parentId", targetParentId);
        }

        SyncData syncData = new SyncData(sourceCategory);
        if (!create(syncData, targetOrganization, null))
            System.out.println("error: failed to update category in organization: " + targetOrganization.name());
    }

    /**
     * Updates a category inside the target organization with parent id.
     * @param targetParentId the parent id of the target
     * @param sourceCategory the date to update to
     * @param targetCategory the target category which will be updated
     * @param targetOrganization the organization where the category will be updated
     */
    private void updateCategory(int targetParentId, JsonObject sourceCategory, JsonObject targetCategory, Organization targetOrganization) {
        if (has(sourceCategory, "parentId")) {
            sourceCategory.addProperty("parentId", targetParentId);
        }

        SyncData syncData = new SyncData(sourceCategory, targetCategory);
        if (!update(syncData, targetOrganization, null))
            System.out.println("error: failed to update category in organization: " + targetOrganization.name());
    }

    /**
     * Prepare filter for organization specific shenanigans <br>
     * Searching the correct category id and saving it to alpha/beta filter
     * @param originalFilter original filter from command line
     * @param alphaFilter beta organization specific filter
     * @param betaFilter alpha organization specific filter
     * @return false if preparation was successful
     */
    protected boolean prepareFilter(HashMap<String, String> originalFilter, HashMap<String, String> alphaFilter, HashMap<String, String> betaFilter) {
        if (originalFilter != null && originalFilter.containsKey("category")) {
            JsonObject alphaCategory = equivalent(originalFilter.get("category"), list(alphaOrganization));
            JsonObject betaCategory = equivalent(originalFilter.get("category"), list(betaOrganization));

            if (isValid(alphaCategory) && isValid(betaCategory)) {
                alphaFilter.putAll(originalFilter);
                betaFilter.putAll(originalFilter);

                alphaFilter.put("categoryId", Integer.toString(alphaCategory.get("id").getAsInt()));
                betaFilter.put("categoryId", Integer.toString(betaCategory.get("id").getAsInt()));

                alphaFilter.remove("category");
                betaFilter.remove("category");

                return true;
            }
        }
        return false;
    }

    @Override
    public boolean isValid(JsonObject category) {
        return has(category, "id") && has(category, "name") &&
                has(category, "lastUpdated");
    }

    /**
     * Get category tree.
     * @param id tree parent, use -1 if you want to get the entire category tree
     * @param organization the organization to get category tree from
     * @return a json array of categories, categories containing child categories
     */
    public JsonArray tree(int id, Organization organization) {
        HashMap<String, String> params = new HashMap<>();
        if (id != -1)
            params.put("id", Integer.toString(id));

        HttpResponse<String> response = http.get(organization, treeEndpoint, params, HttpResponse.BodyHandlers.ofString());
        if (response != null && response.statusCode() == 200) {
            JsonObject body = gson.fromJson(response.body(), JsonObject.class);
            return has(body, "data") ? body.get("data").getAsJsonArray() : null;
        }
        return null;
    }

    /**
     * Check if category is a leaf.
     * @param category the category to check
     * @return true if category is a leaf
     */
    public boolean isLeaf(JsonObject category) {
        return category.get("leaf").getAsBoolean();
    }

    @Override
    protected void vitalInfo(StringBuilder info, JsonObject category) {
        if (has(category, "id"))
            info.append(" id: ").append(category.get("id"));
        if (has(category, "name"))
            info.append(" name: ").append(category.get("name"));
        if (has(category, "parentId"))
            info.append(" parentId: ").append(category.get("parentId"));
        if (has(category, "leaf"))
            info.append(" leaf: ").append(category.get("leaf"));
        if (has(category, "data"))
            info.append(" data: ").append(category.get("data"));
    }
}