package com.cashctrl.orgsync;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;

import java.time.Duration;
import java.util.HashMap;
import java.util.List;

/**
 * Article API synchronize, list, read, create and update articles.
 * @author Silian Barlogis
 * @see SyncObject
 */
public class Article extends Attachment {

    /**
     * Category API for synchronizing article categories.
     */
    private final ArticleCategory articleCategory;

    /**
     * Unit API for synchronizing units.
     */
    private final Unit unit;

    /**
     * Currency API for synchronizing currencies.
     */
    private final Currency currency;

    /**
     * Article constructor, initializes all needed endpoints for article synchronization.
     * @param alphaOrganization alpha organization
     * @param betaOrganization beta organization
     */
    public Article(Organization alphaOrganization, Organization betaOrganization) {
        super(alphaOrganization, betaOrganization, "/api/v1/inventory/article/list.json", "/api/v1/inventory/article/read.json",
                "/api/v1/inventory/article/create.json", "/api/v1/inventory/article/update.json", "/api/v1/inventory/article/update_attachments.json",
                "nr", "");

        articleCategory = new ArticleCategory(alphaOrganization, betaOrganization);
        unit = new Unit(alphaOrganization, betaOrganization);
        currency = new Currency(alphaOrganization, betaOrganization);
    }

    @SuppressWarnings("DuplicatedCode")
    @Override
    public State synchronize() {
        http.setRequestDuration(Duration.ofSeconds(20));
        State state = State.Success;

        System.out.println("--------- synchronize: article categories ---------");
        if (filter != null && filter.containsKey("category")) {
            articleCategory.setFilter(filter);
        }

        switch (articleCategory.synchronize()) {
            case Success -> System.out.println("success: article categories synchronized");
            case NoChange -> System.out.println("no change: there are no article categories to synchronize.");
            case Error -> {
                System.out.println("error: failed to synchronize article categories!");
                state = State.Error;
            }
        }

        System.out.println("--------- synchronize: units ---------");
        switch (unit.synchronize()) {
            case Success -> System.out.println("success: units synchronized");
            case NoChange -> System.out.println("no change: there are no units to synchronize.");
            case Error -> {
                System.out.println("error: failed to synchronize units!");
                state = State.Error;
            }
        }

        System.out.println("--------- synchronize: currencies ---------");
        switch (currency.synchronize()) {
            case Success -> System.out.println("success: currencies synchronized");
            case NoChange -> System.out.println("no change: there are no currencies to synchronize.");
            case Error -> {
                System.out.println("error: failed to synchronize currencies!");
                state = State.Error;
            }
        }

        System.out.println("--------- synchronize: articles ---------");
        HashMap<String, String> alphaFilter = new HashMap<>();
        HashMap<String, String> betaFilter = new HashMap<>();

        if (filter != null) {
            if (filter.containsKey("category")) {
                if (!articleCategory.prepareFilter(filter, alphaFilter, betaFilter)) {
                    System.out.println("error: failed to prepare category filter");
                    return State.Error;
                }
            } else {
                alphaFilter.putAll(filter);
                betaFilter.putAll(filter);
            }
        }

        JsonArray alphaArticles = list(alphaFilter, alphaOrganization);
        JsonArray betaArticles = list(betaFilter, betaOrganization);

        if (alphaArticles == null || betaArticles == null)
            return State.Error;
        else if (alphaArticles.isEmpty() && betaArticles.isEmpty())
            return State.NoChange;

        for (int alphaIdx = 0; alphaIdx < alphaArticles.size(); alphaIdx++) {
            JsonObject articleAlpha = alphaArticles.get(alphaIdx).getAsJsonObject();
            if (!isValid(articleAlpha))
                continue;

            boolean existsInBeta = false;

            for (int betaIdx = 0; betaIdx < betaArticles.size(); betaIdx++) {
                JsonObject articleBeta = betaArticles.get(betaIdx).getAsJsonObject();
                if (!isValid(articleBeta))
                    continue;

                // article number match?
                if (equals(articleAlpha, articleBeta)) {
                    existsInBeta = true;

                    long alphaTime = lastUpdated(articleAlpha);
                    long betaTime = lastUpdated(articleBeta);

                    if (alphaTime != betaTime && (alphaTime > alphaOrganization.lastSynchronized() ||
                            betaTime > betaOrganization.lastSynchronized())) {
                        JsonObject articleAlphaDetails = read(articleAlpha.get("id").getAsInt(), alphaOrganization);
                        JsonObject articleBetaDetails = read(articleBeta.get("id").getAsInt(), betaOrganization);

                        if (alphaTime > betaTime && alphaTime > alphaOrganization.lastSynchronized()) {
                            SyncData syncData = new SyncData(articleAlpha, articleBeta, articleAlphaDetails, articleBetaDetails);
                            if (!updateArticle(syncData, alphaOrganization, betaOrganization)) {
                                state = State.Error;
                            }
                        } else if (betaTime > alphaTime && betaTime > betaOrganization.lastSynchronized()) {
                            SyncData syncData = new SyncData(articleBeta, articleAlpha, articleBetaDetails, articleAlphaDetails);
                            if (!updateArticle(syncData, betaOrganization, alphaOrganization)) {
                                state = State.Error;
                            }
                        }
                    }
                }
            }

            if (!existsInBeta) {
                JsonObject articleAlphaDetails = read(articleAlpha.get("id").getAsInt(), alphaOrganization);
                SyncData syncData = new SyncData(articleAlpha, new JsonObject(), articleAlphaDetails, new JsonObject());
                if (!createArticle(syncData, alphaOrganization, betaOrganization)) {
                    state = State.Error;
                }
            }
        }

        // create beta article in organization alpha
        for (int betaIdx = 0; betaIdx < betaArticles.size(); betaIdx++) {
            JsonObject articleBeta = betaArticles.get(betaIdx).getAsJsonObject();
            if (!isValid(articleBeta))
                continue;

            boolean existsInAlpha = false;
            for (int alphaIdx = 0; alphaIdx < alphaArticles.size(); alphaIdx++) {

                // check if the article is valid and has the needed fields to compare
                JsonObject articleAlpha = alphaArticles.get(alphaIdx).getAsJsonObject();
                if (!isValid(articleAlpha))
                    continue;
                if (equals(articleAlpha, articleBeta))
                    existsInAlpha = true;
            }

            if (!existsInAlpha) {
                JsonObject articleBetaDetails = read(articleBeta.get("id").getAsInt(), betaOrganization);
                SyncData syncData = new SyncData(articleBeta, new JsonObject(), articleBetaDetails, new JsonObject());
                if (!createArticle(syncData, betaOrganization, alphaOrganization)) {
                    state = State.Error;
                }
            }
        }
        return state;
    }

    @Override
    protected JsonArray list(HashMap<String, String> filter, Organization organization) {
        if (filter != null && !filter.containsKey("limit")) {
            filter.put("limit", Integer.toString(limit));
        }

        return super.list(filter, organization);
    }

    /**
     * Creates an article in target organization with syncData.
     * @param syncData synchronization data, source is used for creating
     * @param sourceOrganization the source organisation where the date comes from
     * @param targetOrganization the target organisation where the new article is created with synchronization data
     */
    private boolean createArticle(SyncData syncData, Organization sourceOrganization, Organization targetOrganization) {
        updateDependencies(syncData, sourceOrganization, targetOrganization);
        if (!create(syncData, targetOrganization, sourceOrganization)) {
            System.out.println("error: failed to create article in organization: " + targetOrganization.name());
            return false;
        }
        updateAttachments(syncData, targetOrganization);
        return true;
    }

    /**
     * Updates an article in targetOrganization with syncData.
     * @param syncData synchronization data, source is used for updating the target
     * @param sourceOrganization the source organisation where the data comes from
     * @param targetOrganization the target organisation where the target article is located
     */
    private boolean updateArticle(SyncData syncData, Organization sourceOrganization, Organization targetOrganization) {
        updateDependencies(syncData, sourceOrganization, targetOrganization);
        if (!update(syncData, targetOrganization, sourceOrganization)) {
            System.out.println("error: failed to update article in organization: " + targetOrganization.name());
            return false;
        }
        updateAttachments(syncData, targetOrganization);
        return true;
    }

    /**
     * Update article dependencies like categoryIds, unitIds & currencyIds
     * @param syncData synchronization data
     * @param sourceOrganization the source organisation
     * @param targetOrganization the target organisation
     */
    private void updateDependencies(SyncData syncData, Organization sourceOrganization, Organization targetOrganization) {
        if (has(syncData.source(), "categoryId"))
            articleCategory.fixDependency(syncData, sourceOrganization, targetOrganization);

        if (has(syncData.source(), "unitId"))
            unit.fixDependency(syncData, sourceOrganization, targetOrganization);

        if (has(syncData.source(), "currencyId"))
            currency.fixDependency(syncData, sourceOrganization, targetOrganization);
    }

    @Override
    protected HashMap<String, String> parametrize(SyncData syncData) {
        JsonObject article = syncData.source();

        var fields = List.of(
                // TEXT
                new Parameter("name", article, DataType.STRING),
                new Parameter("binLocation", article, DataType.STRING),
                new Parameter("custom", article, DataType.STRING),
                new Parameter("description", article, DataType.STRING),
                new Parameter("notes", article, DataType.STRING),
                new Parameter("nr", article, DataType.STRING),

                // NUMBER (ints)
                new Parameter("categoryId", article, DataType.INT),
                new Parameter("currencyId", article, DataType.INT),
                new Parameter("unitId", article, DataType.INT),
                // TODO new Parameter("locationId",        article, DataType.INT),
                // TODO new Parameter("sequenceNumberId",  article, DataType.INT),

                // NUMBER (decimals)
                new Parameter("lastPurchasePrice", article, DataType.FLOAT),
                new Parameter("salesPrice", article, DataType.FLOAT),
                new Parameter("maxStock", article, DataType.DOUBLE),
                new Parameter("minStock", article, DataType.DOUBLE),
                new Parameter("stock", article, DataType.DOUBLE),

                // BOOLEAN
                new Parameter("isInactive", article, DataType.BOOLEAN),
                new Parameter("isPurchasePriceGross", article, DataType.BOOLEAN),
                new Parameter("isSalesPriceGross", article, DataType.BOOLEAN),
                new Parameter("isStockArticle", article, DataType.BOOLEAN)
        );

        return parametrizeParams(fields);
    }

    @Override
    public boolean isValid(JsonObject article) {
        return has(article, "id") && has(article, "name") &&
                has(article, "lastUpdated");
    }

    @Override
    protected void vitalInfo(StringBuilder info, JsonObject article) {
        if (has(article, "id"))
            info.append(" id: ").append(article.get("id"));
        if (has(article, "name"))
            info.append(" name; ").append(article.get("name").getAsString());
        if (has(article, "binLocation"))
            info.append(" binLocation: ").append(article.get("binLocation").getAsString());
        if (has(article, "categoryId"))
            info.append(" categoryId: ").append(article.get("categoryId").getAsInt());
        if (has(article, "currencyId"))
            info.append(" currencyId: ").append(article.get("currencyId").getAsInt());
        if (has(article, "locationId"))
            info.append(" locationId: ").append(article.get("locationId").getAsInt());
        if (has(article, "nr"))
            info.append(" nr: ").append(article.get("nr").getAsString());
        if (has(article, "unitId"))
            info.append(" unitId ").append(article.get("unitId").getAsInt());
    }
}