package com.cashctrl.orgsync;

import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;

import java.net.http.HttpResponse;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.List;

/**
 * SyncObject abstract class implement functions for synchronization. <br>
 * Functions: e.g. default synchronize, list, read, create and update etc. <br>
 * To send requests and receive an answer <br>
 * {@link java.net.http.HttpClient}, {@link java.net.http.HttpRequest} and {@link java.net.http.HttpResponse} is used. <br>
 * To read the answer from the CashCtrl API {@link com.google.gson} is used for parsing the JSON. <br>
 * <br>
 * <b>List synchronization</b><br>
 * Synchronizes two lists of object this is the default synchronization method, which will work for most objects.
 * Extend from {@link SyncObject} if you would like to synchronize objects with list synchronization.
 * If the object has no dependencies like a categoryId or titleId, then you can simply use the default synchronize implementation.
 * Take a look at {@link com.cashctrl.orgsync.Currency} or {@link com.cashctrl.orgsync.Title} for an example.
 * If the object has said dependencies you need to overwrite the synchronize function.
 * For an example take a look at {@link com.cashctrl.orgsync.Person#synchronize()} or {@link com.cashctrl.orgsync.Article#synchronize()}. <br>
 * <br>
 * <b>Tree synchronization</b><br>
 * Synchronizes a tree of objects this is a recursive method and is used for synchronizing categories.
 * Extend from {@link com.cashctrl.orgsync.Category} if you would like to use tree synchronization.
 * For an example take a look at {@link com.cashctrl.orgsync.Category#synchronize()} and {@link com.cashctrl.orgsync.ArticleCategory} or {@link com.cashctrl.orgsync.PersonCategory}.
 * @author Silian Barlogis
 * @see com.cashctrl.orgsync.Category
 * @see com.cashctrl.orgsync.HttpMethods
 * @see com.cashctrl.orgsync.Organization
 */
public abstract class SyncObject {
    /**
     * Dataformat for parsing the date, used for comparing which object is more up to date.
     */
    protected final SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    /**
     * HttpMethods instance for POST and GET requests.
     */
    protected final HttpMethods http = HttpMethods.instance();
    /**
     * GSON instance for parsing JSON.
     */
    protected final Gson gson = new Gson();
    /**
     * Alpha and Beta organization.
     */
    protected final Organization alphaOrganization;
    protected final Organization betaOrganization;
    /**
     * List, read, create and update endpoints.
     */
    protected final String listEndpoint;
    protected final String readEndpoint;
    protected final String createEndpoint;
    protected final String updateEndpoint;
    /**
     * The name of the property for equality test for example (category name) "name" or (personal number) "nr".
     */
    protected final String equalProperty;
    /**
     * The name of the dependency property for example the titleId in a person which points to a title.
     */
    protected final String dependencyProperty;
    /**
     * Custom record limit
     */
    protected final int limit = 99999;
    /**
     * Filter used for filtering organization objects applied in {@link #list(HashMap, Organization)}.
     */
    protected HashMap<String, String> filter = null;

    /**
     * SyncObject constructor
     * @param alphaOrganization the alpha organization
     * @param betaOrganization the beta organization
     * @param listEndpoint endpoint for list action
     * @param readEndpoint endpoint for read action
     * @param createEndpoint endpoint for create action
     * @param updateEndpoint endpoint for update action
     * @param equalProperty name of the property for equality test
     * @param dependencyProperty name of the dependency property
     */
    protected SyncObject(Organization alphaOrganization, Organization betaOrganization, String listEndpoint, String readEndpoint, String createEndpoint, String updateEndpoint,
                         String equalProperty, String dependencyProperty) {
        this.alphaOrganization = alphaOrganization;
        this.betaOrganization = betaOrganization;
        this.listEndpoint = listEndpoint;
        this.readEndpoint = readEndpoint;
        this.createEndpoint = createEndpoint;
        this.updateEndpoint = updateEndpoint;
        this.equalProperty = equalProperty;
        this.dependencyProperty = dependencyProperty;
    }

    /**
     * Default synchronization using list synchronization.
     * @return return success, no change or error
     */
    public State synchronize() {
        JsonArray alphaObjects = list(alphaOrganization);
        JsonArray betaObjects = list(betaOrganization);

        if (alphaObjects == null || betaObjects == null)
            return State.Error;
        else if (alphaObjects.isEmpty() && betaObjects.isEmpty())
            return State.NoChange;

        for (int alphaIdx = 0; alphaIdx < alphaObjects.size(); alphaIdx++) {

            JsonObject alphaObject = alphaObjects.get(alphaIdx).getAsJsonObject();
            if (!isValid(alphaObject))
                continue;

            boolean existsInBeta = false;

            for (int betaIdx = 0; betaIdx < betaObjects.size(); betaIdx++) {
                JsonObject betaObject = betaObjects.get(betaIdx).getAsJsonObject();
                if (!isValid(betaObject))
                    continue;

                // update if is equal
                if (equals(alphaObject, betaObject)) {
                    existsInBeta = true;

                    long alphaTime = lastUpdated(alphaObject);
                    long betaTime = lastUpdated(betaObject);

                    if (alphaTime != betaTime) {
                        if (alphaTime > betaTime && alphaTime > alphaOrganization.lastSynchronized()) {
                            SyncData syncData = new SyncData(alphaObject, betaObject);
                            if (!update(syncData, betaOrganization, alphaOrganization)) {
                                System.out.println("error: failed to update in organization: " + betaOrganization.name());
                                return State.Error;
                            }
                        } else if (betaTime > alphaTime && betaTime > betaOrganization.lastSynchronized()) {
                            SyncData syncData = new SyncData(betaObject, alphaObject);
                            if (!update(syncData, alphaOrganization, betaOrganization)) {
                                System.out.println("error: failed to update in organization: " + alphaOrganization.name());
                                return State.Error;
                            }
                        }
                    }
                }
            }

            // create alpha in organization beta if it does not exist
            if (!existsInBeta) {
                SyncData syncData = new SyncData(alphaObject);
                if (!create(syncData, betaOrganization, alphaOrganization)) {
                    System.out.println("error: failed to create in organization: " + betaOrganization.name());
                    return State.Error;
                }
            }
        }

        // create beta in organization alpha if it does not exist
        for (int betaIdx = 0; betaIdx < betaObjects.size(); betaIdx++) {
            JsonObject betaObject = betaObjects.get(betaIdx).getAsJsonObject();
            boolean existsInAlpha = false;

            for (int alphaIdx = 0; alphaIdx < alphaObjects.size(); alphaIdx++) {
                JsonObject alphaObject = alphaObjects.get(alphaIdx).getAsJsonObject();
                if (equals(alphaObject, betaObject))
                    existsInAlpha = true;
            }

            if (!existsInAlpha) {
                SyncData syncData = new SyncData(betaObject);
                if (!create(syncData, alphaOrganization, betaOrganization)) {
                    System.out.println("error: failed to create in organization: " + alphaOrganization.name());
                    return State.Error;
                }
            }
        }
        return State.Success;
    }

    public void setFilter(HashMap<String, String> filter) {
        this.filter = filter;
    }

    /**
     * list data in the given organization.
     * @param organization the organization to list from
     * @return response data as json array
     */
    protected JsonArray list(Organization organization) {
        return list(new HashMap<>(), organization);
    }

    /**
     * List data in the given organization.
     * @param filter filter for listing
     * @param organization the organization to list from
     * @return response data as json array
     */
    protected JsonArray list(HashMap<String, String> filter, Organization organization) {
        HttpResponse<String> response = http.get(organization, listEndpoint, filter, HttpResponse.BodyHandlers.ofString());
        if (response != null && response.statusCode() == 200) {
            JsonObject body = gson.fromJson(response.body(), JsonObject.class);
            if (has(body, "data")) {
                return body.get("data").getAsJsonArray();
            }
        }
        return null;
    }

    /**
     * Read the object with id.
     * @param id id of the object
     * @param organization the organization to list from
     * @return response data as json array
     */
    protected JsonObject read(int id, Organization organization) {
        HashMap<String, String> parameter = new HashMap<>();
        parameter.put("id", Integer.toString(id));

        HttpResponse<String> response = http.get(organization, readEndpoint, parameter, HttpResponse.BodyHandlers.ofString());
        if (response != null && response.statusCode() == 200) {
            // read the response and return the detailed information
            JsonObject body = gson.fromJson(response.body(), JsonObject.class);
            return has(body, "data") ? body.get("data").getAsJsonObject() : null;
        }
        return null;
    }

    /**
     * Create the object with the given data.
     * @param syncData data for creation
     * @param targetOrganization the organization where the object should get created in
     * @param sourceOrganization the source organization where the source is located
     * @return response data as json array
     */
    protected boolean create(SyncData syncData, Organization targetOrganization, Organization sourceOrganization) {
        HashMap<String, String> data = parametrize(syncData);
        HttpResponse<String> response = http.post(targetOrganization, createEndpoint, data, HttpResponse.BodyHandlers.ofString());
        if (checkResponse(response, syncData, "create")) {
            syncData.target().addProperty("id", insertId(response));
            return true;
        }

        return false;
    }

    /**
     * Update a single object inside the organization.
     * @param syncData source is data for updating and target is the update target
     * @param targetOrganization the organization where update object is located
     * @param sourceOrganization the source organization where the source is located
     * @return true if successful
     */
    protected boolean update(SyncData syncData, Organization targetOrganization, Organization sourceOrganization) {
        if (has(syncData.target(), "id")) {

            HashMap<String, String> data = parametrize(syncData);
            data.put("id", Integer.toString(syncData.target().get("id").getAsInt()));

            HttpResponse<String> response = http.post(targetOrganization, updateEndpoint, data, HttpResponse.BodyHandlers.ofString());
            return checkResponse(response, syncData, "update");
        }

        System.out.println("error: the target has no id or is null");
        return false;
    }

    /**
     * check response if any errors occurred on the update or creation process.
     * @param response http response from a post request
     * @param syncData synchronization data
     * @param action the action name
     * @return return true if successful
     */
    protected boolean checkResponse(HttpResponse<String> response, SyncData syncData, String action) {
        if (response == null || response.statusCode() != 200)
            return false;

        JsonObject jsonResponse = gson.fromJson(response.body(), JsonObject.class);
        if (has(jsonResponse, "success")) {
            // update was successfully
            if (jsonResponse.get("success").getAsBoolean())
                return true;

            if (syncData != null) {
                System.out.println("error subject: " + getInfo(syncData, action));
            }

            // check for validation errors
            if (has(jsonResponse, "errors")) {
                JsonArray errors = jsonResponse.get("errors").getAsJsonArray();
                for (int idx = 0; idx < errors.size(); idx++) {
                    JsonObject error = errors.get(idx).getAsJsonObject();
                    System.out.println("field: " + (has(error, "field") ? error.get("field").getAsString() : "null"));
                    System.out.println("message: " + (has(error, "message") ? error.get("message").getAsString() : "null"));
                }
            }

            //check for other messages
            if (has(jsonResponse, "message"))
                System.out.println("message: " + jsonResponse.get("message").getAsString());
        }

        return false;
    }

    /**
     * Check if a json object is not null, has the property and is not json-null.
     * @param object the object that will be validated
     * @param property the has property
     * @return true if object is valid and has property which is not json null
     */
    protected boolean has(JsonObject object, String property) {
        return object != null && object.has(property) && !object.get(property).isJsonNull();
    }

    /**
     * Get the insert id of a created object.
     * @param response the response with body
     * @return the insert id, return -1 if failed to get insert id
     */
    protected int insertId(HttpResponse<String> response) {
        if (response == null || response.statusCode() != 200)
            return -1;

        JsonObject jsonResponse = gson.fromJson(response.body(), JsonObject.class);
        return has(jsonResponse, "insertId") ? jsonResponse.get("insertId").getAsInt() : -1;
    }

    /**
     * Parse the lastUpdated field and get time as long.
     * @param object to get the last updated from
     * @return last updated time as long
     */
    protected long lastUpdated(JsonObject object) {
        try {
            String timeStr = object.get("lastUpdated").getAsString();
            return dateFormat.parse(timeStr).getTime();
        } catch (ParseException exception) {
            System.out.println("error: failed to parse date: " + exception);
        }

        return 0;
    }

    /**
     * Get the last edited record time in milliseconds
     * @param organization organization
     * @return last edited in milliseconds
     */
    public long getLastEdited(Organization organization ) {
        HashMap<String, String> filter = new HashMap<>();
        filter.put("sort", "lastUpdated");
        filter.put("dir", "DESC");
        filter.put("limit", "1");

        JsonArray records = list(filter, organization);
        if (records != null && !records.isEmpty()) {
            JsonObject obj = records.get(0).getAsJsonObject();
            if (obj != null && !obj.isJsonNull()) {
                return lastUpdated(obj);
            }
        }

        if (records != null && records.isEmpty()) {
            return new Date().getTime();
        }

        System.out.println("error: failed to get last edited " + this + " in " + organization.name());
        return 0;
    }

    /**
     * Finds the equivalent object inside pool.
     * @param searchObject the search object
     * @param pool the search pool
     * @return the equivalent object in pool or null if not found
     */
    protected JsonObject equivalent(JsonObject searchObject, JsonArray pool) {
        for (int idx = 0; idx < pool.size(); idx++) {
            JsonObject obj = pool.get(idx).getAsJsonObject();
            if (equals(searchObject, obj))
                return obj;
        }
        return null;
    }

    /**
     * Finds the equivalent object inside pool.
     * @param searchValue the search value must be an equality value like person "nr" or category "name"
     * @param pool the search pool
     * @return the equivalent object in pool or null if not found
     */
    protected JsonObject equivalent(String searchValue, JsonArray pool) {
        for (int idx = 0; idx < pool.size(); idx++) {
            JsonObject obj = pool.get(idx).getAsJsonObject();
            if (has(obj, equalProperty) && obj.get(equalProperty).getAsString().equals(searchValue)) {
                return obj;
            }
        }
        return null;
    }

    /**
     * Updates the dependency property inside synData (source) e.g. <br>
     * Assuming synData is a person and this function is called on a
     * {@link com.cashctrl.orgsync.Title} object, this function will try to find the
     * titleIds associated Title object inside the target organisation and update the source accordingly.
     * @param syncData synchronization data
     * @param sourceOrg source organization
     * @param targetOrg source organization
     */
    protected void fixDependency(SyncData syncData, Organization sourceOrg, Organization targetOrg) {
        JsonObject source = syncData.source();
        JsonObject sourceObject = read(source.get(dependencyProperty).getAsInt(), sourceOrg);
        JsonArray targetObjects = list(targetOrg);

        // get equivalent object in target organization
        if (has(sourceObject, equalProperty) && targetObjects != null) {
            JsonObject equivalentObject = equivalent(sourceObject, targetObjects);
            if (equivalentObject != null) {
                // update the dependency with the correct target object dependency id
                source.addProperty(dependencyProperty, equivalentObject.get("id").getAsInt());
                return;
            }
        }

        // error output
        if (targetObjects == null || targetObjects.isEmpty())
            System.out.println("error: no objects in organization: " + targetOrg.url());

        System.out.println("error: failed to fix dependency " + dependencyProperty + " in: \n" + getInfo(syncData, "") +
                (has(sourceObject, equalProperty) ? " \nCouldn't find: " + sourceObject.get(equalProperty) : ""));
        source.remove(dependencyProperty);
    }

    /**
     * Test if object is equal according to the equal property.
     * @param alphaObject alpha object to test
     * @param betaObject beta object to test
     * @return true if objects are equal
     */
    protected boolean equals(JsonObject alphaObject, JsonObject betaObject) {
        return has(alphaObject, equalProperty) && has(betaObject, equalProperty) &&
                alphaObject.get(equalProperty).equals(betaObject.get(equalProperty));
    }

    /**
     * Transforms a object into a kay / value layout.
     * @param syncData synchronization data to turn into key / value output
     * @return synchronization data as a kay / value layout
     */
    protected abstract HashMap<String, String> parametrize(SyncData syncData);

    /**
     * Transform a list of parameters into a kay / value layout.
     * @param fields parameters
     * @return data as a kay / value layout
     */
    protected HashMap<String, String> parametrizeParams(List<Parameter> fields) {
        HashMap<String, String> data = new HashMap<>();
        for (Parameter f : fields) {
            if (f.source() != null && has(f.source(), f.name())) {
                switch (f.type()) {
                    case STRING -> data.put(f.name(), f.source().get(f.name()).getAsString());
                    case INT -> data.put(f.name(), Integer.toString(f.source().get(f.name()).getAsInt()));
                    case FLOAT -> data.put(f.name(), Float.toString(f.source().get(f.name()).getAsFloat()));
                    case DOUBLE -> data.put(f.name(), Double.toString(f.source().get(f.name()).getAsDouble()));
                    case BOOLEAN -> data.put(f.name(), Boolean.toString(f.source().get(f.name()).getAsBoolean()));
                    case JSON_ARRAY -> data.put(f.name(), gson.toJson(f.source().get(f.name()).getAsJsonArray()));
                }
            }
        }
        return data;
    }

    /**
     * Get information about the current object.
     * @param syncData synchronization data to get the information from
     * @param action the action that was executed
     * @return information as string
     */
    protected String getInfo(SyncData syncData, String action) {
        StringBuilder info = new StringBuilder();
        if (!action.isBlank()) {
            info.append("ACTION: ").append(action).append("\n");
        }

        if (syncData.source() != null) {
            info.append("SOURCE: ");
            vitalInfo(info, syncData.source());
            info.append("\n");
        } else {
            info.append("SOURCE is null\n");
        }

        if (syncData.target() != null) {
            info.append("TARGET: ");
            vitalInfo(info, syncData.target());
        } else {
            info.append("TARGET is null");
        }

        return info.toString();
    }

    /**
     * Validate object for synchronization.
     * @param object object to be validated
     * @return true if object is valid
     */
    protected abstract boolean isValid(JsonObject object);

    /**
     * Add vital information.
     * @param info stringBuilder to add information
     * @param subject to get vital information from
     */
    protected abstract void vitalInfo(StringBuilder info, JsonObject subject);

    /**
     * State for defining the process state. <br>
     * Success, success with no critical errors, doesn't mean that every object could have been synchronized.
     * NoChange, there is nothing to synchronize.
     * Error, a critical error has occurred so that the process can not continue.
     */
    public enum State {
        Success,
        NoChange,
        Error,
    }

    /**
     * Datatype of parameter {@link Parameter}
     */
    protected enum DataType {
        STRING,
        INT,
        FLOAT,
        DOUBLE,
        BOOLEAN,
        JSON_ARRAY
    }

    /**
     * Parameter record storage for parametrize
     * @param name name of the parameter
     * @param source data
     * @param type type of the data {@link DataType}
     */
    protected record Parameter(String name, JsonObject source, DataType type) {
    }
}