package com.cashctrl.orgsync;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;

import java.time.Duration;
import java.util.HashMap;
import java.util.List;

/**
 * Person API synchronize, list, read, create and update people.
 * @author Silian Barlogis
 * @see SyncObject
 */
public class Person extends Attachment {
    /**
     * Category API for synchronizing person category.
     */
    private final PersonCategory personCategory;
    /**
     * Title API for synchronizing person titles.
     */
    private final Title title;

    /**
     * Person constructor, initializes all needed endpoints for person synchronization.
     * @param alphaOrganization alpha organization
     * @param betaOrganization beta organization
     */
    public Person(Organization alphaOrganization, Organization betaOrganization) {
        super(alphaOrganization, betaOrganization, "/api/v1/person/list.json", "/api/v1/person/read.json",
                "/api/v1/person/create.json", "/api/v1/person/update.json", "/api/v1/person/update_attachments.json",
                "nr", "");

        personCategory = new PersonCategory(alphaOrganization, betaOrganization);
        title = new Title(alphaOrganization, betaOrganization);
    }

    @SuppressWarnings("DuplicatedCode")
    @Override
    public State synchronize() {
        http.setRequestDuration(Duration.ofSeconds(20));

        State state = State.Success;

        System.out.println("--------- synchronize: person categories ---------");
        if (filter != null && filter.containsKey("category")) {
            personCategory.setFilter(filter);
        }

        switch (personCategory.synchronize()) {
            case Success -> System.out.println("success: person categories synchronized");
            case NoChange -> System.out.println("no change: there are no person categories to synchronize.");
            case Error -> {
                System.out.println("error: failed to synchronize person categories!");
                state = State.Error;
            }
        }

        System.out.println("--------- synchronize: titles ---------");
        switch (title.synchronize()) {
            case Success -> System.out.println("success: titles synchronized");
            case NoChange -> System.out.println("no change: there are no titles to synchronize.");
            case Error -> {
                System.out.println("error: failed to synchronize titles!");
                state = State.Error;
            }
        }

        System.out.println("--------- synchronize: people ---------");
        HashMap<String, String> alphaFilter = new HashMap<>();
        HashMap<String, String> betaFilter = new HashMap<>();

        if (filter != null) {
            if (filter.containsKey("category")) {
                if (!personCategory.prepareFilter(filter, alphaFilter, betaFilter)) {
                    System.out.println("error: failed to prepare category filter");
                    return State.Error;
                }
            } else {
                alphaFilter.putAll(filter);
                betaFilter.putAll(filter);
            }
        }

        // hierarchical map, key is the subordinate and value is the superior
        HashMap<JsonObject, JsonObject> alphaHierarchicalMap = new HashMap<>();
        HashMap<JsonObject, JsonObject> betaHierarchicalMap = new HashMap<>();

        JsonArray alphaPeople = list(alphaFilter, alphaOrganization);
        JsonArray betaPeople = list(betaFilter, betaOrganization);

        if (alphaPeople == null || betaPeople == null)
            return State.Error;
        else if (alphaPeople.isEmpty() && betaPeople.isEmpty())
            return State.NoChange;

        for (int alphaIdx = 0; alphaIdx < alphaPeople.size(); alphaIdx++) {
            JsonObject personAlpha = alphaPeople.get(alphaIdx).getAsJsonObject();
            if (!isValid(personAlpha))
                continue;

            boolean existsInBeta = false;
            for (int betaIdx = 0; betaIdx < betaPeople.size(); betaIdx++) {
                JsonObject personBeta = betaPeople.get(betaIdx).getAsJsonObject();
                if (!isValid(personBeta))
                    continue;

                if (equals(personAlpha, personBeta)) {
                    existsInBeta = true;

                    long alphaTime = lastUpdated(personAlpha);
                    long betaTime = lastUpdated(personBeta);
                    if (alphaTime != betaTime && (alphaTime > alphaOrganization.lastSynchronized() ||
                            betaTime > betaOrganization.lastSynchronized())) {
                        JsonObject personAlphaDetails = read(personAlpha.get("id").getAsInt(), alphaOrganization);
                        JsonObject personBetaDetails = read(personBeta.get("id").getAsInt(), betaOrganization);

                        if (alphaTime > betaTime && alphaTime > alphaOrganization.lastSynchronized()) {
                            SyncData syncData = new SyncData(personAlpha, personBeta, personAlphaDetails, personBetaDetails);
                            if (!updatePerson(syncData, betaHierarchicalMap, alphaOrganization, betaOrganization)) {
                                state = State.Error;
                            }
                        } else if (betaTime > alphaTime && betaTime > betaOrganization.lastSynchronized()) {
                            SyncData syncData = new SyncData(personBeta, personAlpha, personBetaDetails, personAlphaDetails);
                            if (!updatePerson(syncData, alphaHierarchicalMap, betaOrganization, alphaOrganization)) {
                                state = State.Error;
                            }
                        }
                    }
                }
            }

            // create alpha person in organization beta
            if (!existsInBeta) {
                JsonObject personAlphaDetails = read(personAlpha.get("id").getAsInt(), alphaOrganization);
                SyncData syncData = new SyncData(personAlpha, new JsonObject(), personAlphaDetails, new JsonObject());
                if (!createPerson(syncData, betaHierarchicalMap, alphaOrganization, betaOrganization)) {
                    state = State.Error;
                }
            }
        }

        // create beta person in organization alpha
        for (int betaIdx = 0; betaIdx < betaPeople.size(); betaIdx++) {
            JsonObject personBeta = betaPeople.get(betaIdx).getAsJsonObject();
            if (!isValid(personBeta))
                continue;

            boolean existsInAlpha = false;
            for (int alphaIdx = 0; alphaIdx < alphaPeople.size(); alphaIdx++) {
                JsonObject personAlpha = alphaPeople.get(alphaIdx).getAsJsonObject();
                if (!isValid(personAlpha))
                    continue;
                if (equals(personAlpha, personBeta))
                    existsInAlpha = true;
            }

            if (!existsInAlpha) {
                JsonObject personBetaDetails = read(personBeta.get("id").getAsInt(), betaOrganization);
                SyncData syncData = new SyncData(personBeta, new JsonObject(), personBetaDetails, new JsonObject());
                if (!createPerson(syncData, alphaHierarchicalMap, betaOrganization, alphaOrganization)) {
                    state = State.Error;
                }
            }
        }

        updateSuperiorIds(alphaHierarchicalMap, alphaOrganization);
        updateSuperiorIds(betaHierarchicalMap, betaOrganization);
        return state;
    }

    @Override
    protected HashMap<String, String> parametrize(SyncData syncData) {
        JsonObject person = syncData.source();
        JsonObject personDetails = syncData.sourceDetails();

        var fields = List.of(
                // TEXT
                new Parameter("company", person, DataType.STRING),
                new Parameter("firstName", person, DataType.STRING),
                new Parameter("lastName", person, DataType.STRING),
                new Parameter("altName", person, DataType.STRING),
                new Parameter("bankData", person, DataType.STRING),
                new Parameter("bic", person, DataType.STRING), // kept for compatibility
                new Parameter("color", person, DataType.STRING),
                new Parameter("custom", person, DataType.STRING),
                new Parameter("dateBirth", person, DataType.STRING),
                new Parameter("department", person, DataType.STRING),
                new Parameter("iban", person, DataType.STRING), // kept for compatibility
                new Parameter("industry", person, DataType.STRING),
                new Parameter("language", person, DataType.STRING),
                new Parameter("notes", person, DataType.STRING),
                new Parameter("nr", person, DataType.STRING),
                new Parameter("position", person, DataType.STRING),
                new Parameter("ssn", person, DataType.STRING),
                new Parameter("vatUid", person, DataType.STRING),
                new Parameter("certificateValues", personDetails, DataType.STRING),

                // NUMBER (ints)
                new Parameter("categoryId", person, DataType.INT),
                new Parameter("superiorId", person, DataType.INT),
                new Parameter("titleId", person, DataType.INT),
                // TODO new Parameter("certificateTemplateId",person, DataType.INT),
                // TODO new Parameter("locationId", person, DataType.INT),
                // TODO new Parameter("sequenceNumberId", person, DataType.INT),
                // TODO new Parameter("userId", person, DataType.INT),

                // NUMBER (decimals)
                new Parameter("discountPercentage", person, DataType.FLOAT),

                // BOOLEAN
                new Parameter("isAutoFillResponsiblePerson", person, DataType.BOOLEAN),
                new Parameter("isCustomer", person, DataType.BOOLEAN),
                new Parameter("isEmployee", person, DataType.BOOLEAN),
                new Parameter("isFamily", person, DataType.BOOLEAN),
                new Parameter("isHideCompany", person, DataType.BOOLEAN),
                new Parameter("isHideName", person, DataType.BOOLEAN),
                new Parameter("isInactive", person, DataType.BOOLEAN),
                new Parameter("isInsurance", person, DataType.BOOLEAN),
                new Parameter("isVendor", person, DataType.BOOLEAN),

                // JSON ARRAY
                new Parameter("bankAccounts", personDetails, DataType.JSON_ARRAY),
                new Parameter("children", personDetails, DataType.JSON_ARRAY),
                new Parameter("insuranceContracts", personDetails, DataType.JSON_ARRAY),
                new Parameter("servicePeriods", personDetails, DataType.JSON_ARRAY),
                new Parameter("addresses", personDetails, DataType.JSON_ARRAY),
                new Parameter("contacts", personDetails, DataType.JSON_ARRAY)
        );

        return parametrizeParams(fields);
    }

    @Override
    public boolean isValid(JsonObject person) {
        return has(person, "id") && has(person, "nr") &&
                has(person, "lastUpdated") && (has(person, "firstName") ||
                has(person, "lastName") || has(person, "company"));
    }

    @Override
    protected JsonArray list(HashMap<String, String> filter, Organization organization) {
        if (filter != null && !filter.containsKey("limit")) {
            filter.put("limit", Integer.toString(limit));
        }

        return super.list(filter, organization);
    }

    /**
     * Creates a person in {@code targetOrganization} with {@code syncData}.
     * @param syncData synchronization data, source is used for creating
     * @param hierarchicalMap hierarchical map with subordinates and there superior
     * @param sourceOrganization the source organisation where the date comes from
     * @param targetOrganization the target organisation where the new person is created with synchronization data
     */
    private boolean createPerson(SyncData syncData, HashMap<JsonObject, JsonObject> hierarchicalMap, Organization sourceOrganization, Organization targetOrganization) {
        updateDependencies(syncData, hierarchicalMap, sourceOrganization, targetOrganization);
        if (!create(syncData, targetOrganization, sourceOrganization)) {
            System.out.println("error: failed to create person in organization: " + targetOrganization.name());
            return false;
        }
        updateAttachments(syncData, targetOrganization);
        return true;
    }

    /**
     * Updates a person in targetOrganization with syncData.
     * @param syncData synchronization data, source is used for updating the target
     * @param hierarchicalMap hierarchical map with subordinates and there superior
     * @param sourceOrganization the source organisation where the data comes from
     * @param targetOrganization the target organisation where the target person is located
     */
    private boolean updatePerson(SyncData syncData, HashMap<JsonObject, JsonObject> hierarchicalMap, Organization sourceOrganization, Organization targetOrganization) {
        updateDependencies(syncData, hierarchicalMap, sourceOrganization, targetOrganization);
        if (!update(syncData, targetOrganization, sourceOrganization)) {
            System.out.println("error: failed to update person in organization: " + targetOrganization.name());
            return false;
        }
        updateAttachments(syncData, targetOrganization);
        return true;
    }

    /**
     * Update person dependencies like categoryIds and titleIds
     * @param syncData synchronization data
     * @param hierarchicalMap hierarchical map with subordinates and there superior
     * @param sourceOrganization the source organisation
     * @param targetOrganization the target organisation
     */
    private void updateDependencies(SyncData syncData, HashMap<JsonObject, JsonObject> hierarchicalMap, Organization sourceOrganization, Organization targetOrganization) {
        if (has(syncData.source(), "categoryId"))
            personCategory.fixDependency(syncData, sourceOrganization, targetOrganization);

        if (has(syncData.source(), "titleId"))
            title.fixDependency(syncData, sourceOrganization, targetOrganization);

        if (has(syncData.source(), "superiorId"))
            addHierarchicalEntry(syncData, hierarchicalMap, sourceOrganization);
    }

    /**
     * Adds a hierarchical entry to {@code collection}.
     * @param syncData synchronization data
     * @param collection Hierarchical map, key is the subordinate and value is the superior
     * @param sourceOrganization the source organization
     */
    private void addHierarchicalEntry(SyncData syncData, HashMap<JsonObject, JsonObject> collection, Organization sourceOrganization) {
        JsonObject subordinate = syncData.source();
        int superiorId = subordinate.get("superiorId").getAsInt();
        subordinate.remove("superiorId");

        String subordinateNr = subordinate.get("nr").getAsString();
        JsonObject superior = read(superiorId, sourceOrganization);

        if (isValid(superior)) {
            collection.put(subordinate, superior);
            return;
        }

        System.out.println("error: could not find superior of person with personal number: " + subordinateNr + " superiorId: " + superiorId);
    }

    /**
     * Update superior ids with a hierarchical map
     * @param hierarchicalMap map with superior ids
     * @param organization organization to update superior ids on
     */
    private void updateSuperiorIds(HashMap<JsonObject, JsonObject> hierarchicalMap, Organization organization) {
        hierarchicalMap.forEach((key, value) -> {
            JsonArray peopleArray = list(organization);
            JsonObject subordinate = equivalent(key, peopleArray);
            JsonObject superior = equivalent(value, peopleArray);

            if (subordinate != null && superior != null && isValid(subordinate) && isValid(superior)) {
                subordinate.addProperty("superiorId", superior.get("id").getAsInt());
                SyncData syncData = new SyncData(subordinate, subordinate, new JsonObject(), new JsonObject());
                update(syncData, organization, null);
            }
        });
    }

    @Override
    protected void vitalInfo(StringBuilder info, JsonObject person) {
        if (has(person, "id"))
            info.append(" id: ").append(person.get("id"));
        if (has(person, "nr"))
            info.append(" nr: ").append(person.get("nr"));
        if (has(person, "firstName"))
            info.append(" firstName: ").append(person.get("firstName"));
        if (has(person, "categoryId"))
            info.append(" categoryId: ").append(person.get("categoryId"));
        if (has(person, "titleId"))
            info.append(" titleId: ").append(person.get("titleId"));
        if (has(person, "superiorId"))
            info.append(" superiorId: ").append(person.get("superiorId"));
    }
}