package com.cashctrl.orgsync;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;

import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

/**
 * File API synchronize, list, read, create, update, prepare and persist file.
 * @author Silian Barlogis
 * @see SyncObject
 */
public class File extends SyncObject {

    private static File instance;

    protected final String prepareEndpoint;
    protected final String downloadEndpoint;
    protected final String deleteEndpoint;

    /**
     * File category API for synchronizing file category.
     */
    private final FileCategory fileCategory;
    private FileSyncMode fileSyncMode = FileSyncMode.OnlyAttached;

    /**
     * File constructor, initializes all needed endpoints for file synchronization.
     * @param alphaOrganization the alpha organization
     * @param betaOrganization the beta organization
     */
    protected File(final Organization alphaOrganization, final Organization betaOrganization) {
        super(alphaOrganization, betaOrganization, "/api/v1/file/list.json", "/api/v1/file/read.json",
                "/api/v1/file/create.json", "/api/v1/file/update.json", "name", "");

        prepareEndpoint = "/api/v1/file/prepare.json";
        downloadEndpoint = "/api/v1/file/get";
        deleteEndpoint = "/api/v1/file/delete.json";

        fileCategory = new FileCategory(alphaOrganization, betaOrganization);
    }

    /**
     * Get file instance
     * @return instance of this
     */
    public static File instance() {
        return instance;
    }

    /**
     * Initializes the File instance
     * @param alphaOrganization alpha organization
     * @param betaOrganization beta organization
     * @return this instance
     */
    public static File first(Organization alphaOrganization, Organization betaOrganization) {
        if (instance == null) {
            instance = new File(alphaOrganization, betaOrganization);
        }
        return instance;
    }

    /**
     * Download the file with id in organization.
     * @param fileId the file to download
     * @param organization the organization to download the file from
     * @return the file content in bytes
     */
    protected byte[] download(int fileId, Organization organization) {
        HashMap<String, String> filter = new HashMap<>();
        filter.put("id", Integer.toString(fileId));
        HttpResponse<byte[]> response = http.get(organization, downloadEndpoint, filter,
                HttpResponse.BodyHandlers.ofByteArray());
        if (response != null && response.statusCode() == 200) {
            return response.body();
        }

        System.out.println("error: failed to download file from " + organization.name());
        return null;
    }

    /**
     * Inform CashCtrl of the file names and mime types of the files we are going to upload.
     * @param metadata file metadata like mime type
     * @param organization organisation
     * @return the url to write the file to
     */
    protected JsonObject prepare(JsonObject metadata, Organization organization) {
        if (!has(metadata, "mimeType") || !has(metadata, "name")) {
            System.out.println("error: failed to prepare, critical metadata is missing!");
            return null;
        }

        JsonArray files = new JsonArray();
        JsonObject file = new JsonObject();

        file.addProperty("mimeType", metadata.get("mimeType").getAsString());
        file.addProperty("name", metadata.get("name").getAsString());

        if (has(metadata, "categoryId"))
            file.addProperty("categoryId", Integer.toString(metadata.get("categoryId").getAsInt()));
        if (has(metadata, "size"))
            file.addProperty("size", Integer.toString(metadata.get("size").getAsInt()));

        files.add(file);

        HashMap<String, String> data = new HashMap<>();
        data.put("files", files.toString());
        HttpResponse<String> response = http.post(organization, prepareEndpoint, data, HttpResponse.BodyHandlers.ofString());
        if (!checkResponse(response, null, "prepare file")) {
            return null;
        }

        JsonObject body = gson.fromJson(response.body(), JsonObject.class);
        if (has(body, "data")) {
            JsonArray writeUrls = body.getAsJsonArray("data");
            if (writeUrls != null && !writeUrls.isEmpty()) {
                JsonObject writeUrl = writeUrls.get(0).getAsJsonObject();
                if (has(writeUrl, "fileId") && has(writeUrl, "writeUrl")) {
                    return writeUrl;
                }
            }
        }

        System.out.println("error: failed to prepare file, response contains no write urls");
        return null;
    }

    /**
     * Deletes or moves a file to recycle bin
     * @param fileId the id of the selected entry
     * @param permanent force permanent deletion of file (do not archive first). Possible values: true, false.
     * @param organization organization
     * @return true if successful
     */
    @SuppressWarnings("SameParameterValue")
    protected boolean delete(int fileId, boolean permanent, Organization organization) {
        HashMap<String, String> data = new HashMap<>();
        data.put("ids", Integer.toString(fileId));
        data.put("force", Boolean.toString(permanent));

        HttpResponse<String> response = http.post(organization, deleteEndpoint, data, HttpResponse.BodyHandlers.ofString());
        return checkResponse(response, null, "delete file");
    }

    /**
     * Check if a file is attached to something
     * @param file the file to check
     * @return true if the files is attached to something
     */
    protected boolean isAttached(JsonObject file) {
        return has(file, "isAttached") && file.get("isAttached").getAsBoolean();
    }

    @Override
    public State synchronize() {
        http.setRequestDuration(Duration.ofSeconds(20));

        System.out.println("--------- synchronize: file categories---------");
        switch (fileCategory.synchronize()) {
            case Success -> System.out.println("success: file categories synchronized");
            case NoChange -> System.out.println("no change: there are no file categories to synchronize.");
            case Error -> System.out.println("error: failed to synchronize file categories!");
        }

        http.setRequestDuration(Duration.ofMinutes(2));
        System.out.println("--------- synchronize: files ---------");
        return super.synchronize();
    }

    @Override
    protected JsonArray list(HashMap<String, String> filter, Organization organization) {
        if (filter != null && !filter.containsKey("limit")) {
            filter.put("limit", Integer.toString(limit));
        }

        return super.list(filter, organization);
    }

    @Override
    protected boolean create(SyncData syncData, Organization targetOrganization, Organization sourceOrganization) {
        if (has(syncData.source(), "categoryId"))
            fileCategory.fixDependency(syncData, sourceOrganization, targetOrganization);

        JsonObject source = syncData.source();
        if (fileSyncMode.equals(FileSyncMode.OnlyAttached) && !isAttached(source)) {
            return true;
        }

        JsonObject writeUrl = prepare(source, targetOrganization);
        byte[] file = download(source.get("id").getAsInt(), sourceOrganization);
        if (file == null || writeUrl == null) {
            return false;
        }

        HttpResponse<String> response = http.put(writeUrl.get("writeUrl").getAsString(),
                has(writeUrl, "mimeType") ? writeUrl.get("mimeType").getAsString() : "",
                file, HttpResponse.BodyHandlers.ofString());
        if (response == null || response.statusCode() != 200) {
            System.out.println("error: failed to write file to authenticated url");
            return false;
        }

        SyncData fileData = new SyncData(source);
        HashMap<String, String> data = parametrize(fileData);
        data.put("id", Integer.toString(writeUrl.get("fileId").getAsInt()));
        return checkResponse(http.post(targetOrganization, createEndpoint, data, HttpResponse.BodyHandlers.ofString()),
                fileData, "create file");
    }

    @Override
    protected boolean update(SyncData syncData, Organization targetOrganization, Organization sourceOrganization) {
        JsonObject source = syncData.source();
        JsonObject target = syncData.target();

        if (fileSyncMode.equals(FileSyncMode.OnlyAttached) && !isAttached(source)) {
            return true;
        }

        if (source.get("size").getAsInt() != target.get("size").getAsInt()) {
            byte[] sourceBytes = download(source.get("id").getAsInt(), sourceOrganization);
            byte[] targetBytes = download(target.get("id").getAsInt(), targetOrganization);
            if (sourceBytes == null || targetBytes == null) {
                return false;
            }

            if (!Arrays.equals(sourceBytes, targetBytes)) {
                return delete(target.get("id").getAsInt(), false, targetOrganization) &&
                        create(syncData, targetOrganization, sourceOrganization);
            }
        }

        return super.update(syncData, targetOrganization, sourceOrganization);
    }

    @Override
    protected HashMap<String, String> parametrize(SyncData syncData) {
        JsonObject file = syncData.source();

        var fields = List.of(
                // TEXT
                new Parameter("id", file, DataType.STRING),
                new Parameter("name", file, DataType.STRING),
                new Parameter("custom", file, DataType.STRING),
                new Parameter("description", file, DataType.STRING),
                new Parameter("notes", file, DataType.STRING),

                // NUMBER
                new Parameter("categoryId", file, DataType.INT)
        );

        return parametrizeParams(fields);
    }

    @SuppressWarnings("unused")
    public FileSyncMode getFileSyncMode() {
        return fileSyncMode;
    }

    public void setFileSyncMode(FileSyncMode fileSyncMode) {
        this.fileSyncMode = fileSyncMode;
    }

    @Override
    protected boolean isValid(JsonObject file) {
        return has(file, "id") && has(file, "isAttached") && has(file, "name")
                && has(file, "mimeType") && has(file, "size");
    }

    @Override
    protected void vitalInfo(StringBuilder info, JsonObject file) {
        if (has(file, "id"))
            info.append(" id: ").append(file.get("id"));
        if (has(file, "name"))
            info.append(" name: ").append(file.get("name"));
        if (has(file, "mimeType"))
            info.append(" mimeType: ").append(file.get("mimeType"));
        if (has(file, "isAttached"))
            info.append(" isAttached: ").append(file.get("isAttached"));
        if (has(file, "description"))
            info.append(" description: ").append(file.get("description"));
    }

    /**
     * Available file synchronization modes
     */
    public enum FileSyncMode {
        OnlyAttached,
        All,
        None
    }
}