package com.cashctrl.orgsync;

import org.apache.commons.cli.*;

import java.util.HashMap;

/**
 * Parse command line arguments with {@link org.apache.commons.cli.CommandLineParser}. <br>
 * The following arguments are possible: <br>
 * <b>-p</b> (synchronize people)<br>
 * <b>-a</b> (synchronize articles)<br>
 * <b>-orga</b> (organization alpha)<br>
 * <b>-keya</b> (organization alpha key)<br>
 * <b>-orgb</b> (organization beta)<br>
 * <b>-keyb</b> (organization beta key)<br>
 * <b>-pf</b> (person filter)<br>
 * <b>-af</b> (article filter)
 * @author Silian Barlogis
 */
public class CLI {

    /**
     * The parsed command line arguments.
     */
    private CommandLine cmd;

    /**
     * Parse command line arguments.
     * @param args arguments to parse
     * @return true if successful
     */
    public boolean parseArguments(String[] args) {
        if (args == null || args.length == 0) {
            printHelp();
            return false;
        }

        try {
            CommandLineParser parser = new DefaultParser();
            cmd = parser.parse(getOptions(), args);
            if (cmd != null) {
                return true;
            }
        } catch (ParseException parseException) {
            System.out.print("Failed to parse the arguments! " + parseException);
        }
        return false;
    }

    /**
     * Check if option h or help is present.
     * @return true if either of those are present
     */
    public boolean wannaSeeHelp() {
        return cmd.hasOption("h") || cmd.hasOption("help");
    }

    /**
     * Print the help constructed out of the options using {@link HelpFormatter}.
     */
    public void printHelp() {
        new HelpFormatter().printHelp("CashCtrl - orgsync", getOptions());
    }

    /**
     * Validates whether important arguments are present.
     * @return true if all vital arguments are present
     */
    public boolean validateArguments() {
        if (isArgumentValid("orga") || isArgumentValid("keya") ||
                isArgumentValid("orgb") || isArgumentValid("keyb")) {
            return false;
        }

        if (!cmd.hasOption("p") && !cmd.hasOption("a")) {
            System.out.println("error: you need to set -p (person), -a (article)");
            return false;
        }

        if (cmd.hasOption("fsm")) {
            String fsm = cmd.getOptionValue("fsm");
            if (fsm == null || !fsm.equalsIgnoreCase("none") && !fsm.equalsIgnoreCase("all") &&
                    !fsm.equalsIgnoreCase("onlyAttached")) {
                System.out.println("error: " + fsm + " is not a file synchronization mode use " +
                        "'all', 'none' or 'onlyAttached'");
                return false;
            }
        }

        if (cmd.hasOption("lsd")) {
            String lsd = cmd.getOptionValue("lsd");
            try {
                @SuppressWarnings("unused") long l = Long.parseLong(lsd);
            } catch (Exception e) {
                System.out.println("error: " + lsd + " is not a number!");
                return false;
            }
        }

        return true;
    }

    /**
     * Parse a person <b>(-pf)</b> or article <b>(-af)</b> filter. <br>
     * A filter consist of pairs seperated by a <b>=</b> e.g. <b><i>category=employee</i></b>.<br>
     * It is possible to have multiple filters seperated by a <b>blank space</b> e.g. <b><i>category=employee onlyActive=true</i></b>.
     * @param filter filter to parse
     * @return the parsed filter as HashMap key is the filterName and value its value
     */
    public HashMap<String, String> parseFilter(String filter) {
        if (isArgumentValid(filter))
            return null;

        HashMap<String, String> filters = new HashMap<>();
        String value = cmd.getOptionValue(filter);
        String[] options = value.split(" ");
        for (String option : options) {
            String[] pairs = option.split("=");
            if (pairs.length < 2)
                continue;

            filters.put(pairs[0], pairs[1]);
        }
        return filters;
    }

    /**
     * Check if the option is present.
     * @param option the option to check for
     * @return true if option is present
     */
    public boolean has(String option) {
        return cmd.hasOption(option);
    }

    /**
     * Get the value of option.
     * @param option the option to get the value from
     * @return the value of the specified option
     */
    public String get(String option) {
        return cmd.getOptionValue(option);
    }

    /**
     * Validates if the argument is set and has a subsequent value.
     * @param arg argument to validate
     * @return false if argument is valid
     */
    private boolean isArgumentValid(String arg) {
        if (!cmd.hasOption(arg)) {
            System.out.println("error: " + arg + " is not present.");
            return true;
        }

        String value = cmd.getOptionValue(arg);
        if (value == null || value.isEmpty()) {
            System.out.println("error: " + arg + " has no subsequent value");
            return true;
        }

        return false;
    }

    /**
     * Defines the command line options.
     * @return defined options
     */
    private Options getOptions() {
        Options options = new Options();
        options.addOption("h", "help", false, "display help");
        options.addOption("p", false, "synchronize people");
        options.addOption("a", false, "synchronize articles");
        options.addOption("orga", true, "organization alpha");
        options.addOption("orgb", true, "organization beta");
        options.addOption("keya", true, "key to access organization alpha");
        options.addOption("keyb", true, "key to access organization beta");
        options.addOption("pf", true, "person filter: if you would like" +
                " to synchronize all people which are employees and active use the following:" +
                " \"category=employees onlyActive=true\"");
        options.addOption("af", true, "article filter: if you would like" +
                " to synchronize all articles which are services and active use the following:" +
                " \"category=services onlyActive=true\"");
        options.addOption("fsm", true, "file synchronization mode: possible values 'none', 'all' or 'onlyAttached', defaults to 'onlyAttached'.");
        options.addOption("lsd", true, "last synchronization delta - the time, in milliseconds, that must elapse since the last synchronization and last edited record before it is synchronized again, defaults to 10 seconds.");
        return options;
    }
}