<?php
set_time_limit(600);

// configuration
const API_KEY = 'zjdqcZjsfSYyM4EPG19tHss6X5NIHf0b';
const ORGANIZATION = 'myorg';
const BASE_URL = 'https://' . ORGANIZATION . '.cashctrl.com/api/v1';
const ZIP_NUMERIC_ONLY = true; // remove non-numeric characters from zip
const CATEGORY_ID = null; // import all contacts into this category

$data = ['csv' => ''];
$result = ['success' => '', 'errors' => []];

// get user input, validate and post data
if (input($data)) {
    try {
        parse($data);
        validate($data);
        post($data);
        $result['success'] = 'The contacts have been imported successfully.';
    } catch (Exception $e) {
        if ($e instanceof ValidationException) {
            $result['errors'] = $e->errors;
        }
        if (!empty($e->getMessage())) {
            $result['errors'][] = $e->getMessage();
        }
    }
}

// show result in CLI
if (isCli()) {
    if ($result['success']) {
        print $result['success'] . "\n";
    } else {
        if (!empty($result['errors'])) {
            print implode("\n", $result['errors']) . "\n";
        } else {
            print "Usage: php " . basename(__FILE__) . " [FILE TO IMPORT]\n";
        }
    }
    die();
}

// show form and result in WEB
header("Content-type: text/html; charset=utf-8");
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <title>Contacts import from CSV</title>
    <style>
        * {box-sizing:border-box}
        body {color:#444;background:white;padding:40px 20px}
        body, input, textarea {font:16px/1.5 Helvetica, Arial, sans-serif;}
        main {max-width:600px;margin:0 auto}
        form .row {margin-bottom:15px}
        input, textarea {padding:8px}
        input[type=text], textarea {border:1px solid #CCC;background:white;width:100%}
        .success, .errors {margin-bottom:20px}
        .success {color:green;font-size:21px}
        .errors {color:red}
        .blurry {color:#BBB}
    </style>
</head>
<body>

<main>
    <h1>Contacts import from CSV</h1>
    <p>Organization: <strong><?= ORGANIZATION ?></strong>.cashctrl.com</p>
    <form method="post" action="?" enctype="multipart/form-data">
        <div class="row">
            <p>The following columns are supported (one or more):</p>
            <ul>
                <li>nr &nbsp;<span class="blurry">(person number)</span></li>
                <li>company</li>
                <li>title &nbsp;<span class="blurry">(e.g. "Mr." or "Mrs.")</span></li>
                <li>firstname</li>
                <li>lastname</li>
                <li>address</li>
                <li>zip &nbsp;<span class="blurry">(postal code)</span></li>
                <li>city</li>
                <li>country &nbsp;<span class="blurry">(2- or 3-chars country code)</span></li>
                <li>email_work</li>
                <li>email_private</li>
                <li>phone_work</li>
                <li>phone_private</li>
                <li>customField1, customField2, ...</li>
            </ul>
            <p>Note: The first row should contain the column names. The column order doesn't matter. All unsupported
                columns are ignored.</p>
        </div>
        <div class="success"><?= he($result['success']) ?></div>
        <div class="errors"><?= implode('<br>', $result['errors']) ?></div>
        <div class="row">
            <h2>Upload file</h2>
            <label><input type="file" name="csv"/></label>
        </div>
        <div class="row">
            <input type="submit" value="Import"/>
        </div>
    </form>
</main>

</body>
</html>
<?php

// FUNCTIONS

/**
 * Get input from user (either input via CLI or from WEB form)
 * @param array $data User/form data
 * @return bool User has entered data or not
 */
function input(&$data) {
    global $argc, $argv;
    if (isCli()) {
        if ($argc > 1) {
            $filename = $argv[1];
            if (file_exists($filename)) {
                $data['csv'] = removeUtf8Bom(file_get_contents($filename));
                return true;
            }
        }
    }
    if ($_POST || $_FILES) {
        if (is_uploaded_file($_FILES['csv']['tmp_name'])) {
            $data['csv'] = removeUtf8Bom(file_get_contents($_FILES['csv']['tmp_name']));
        }
        return true;
    }
    return false;
}

/**
 * Parse CSV
 * @param array $data User/form data
 */
function parse(&$data) {
    $csv = &$data['csv'];
    $commas = substr_count($csv, ',');
    $semicolons = substr_count($csv, ';');
    $tabs = substr_count($csv, "\t");
    $delimiter = ',';
    if ($semicolons > $commas) {
        $delimiter = ';';
    }
    if ($tabs > $semicolons) {
        $delimiter = "\t";
    }
    $csv = str_getcsv($csv, "\n");
    foreach ($csv as &$row) {
        $row = str_getcsv($row, $delimiter);
    }
    // create associative array
    array_walk($csv, function(&$a) use ($csv) {
        $a = array_combine($csv[0], $a);
    });
    // remove column header
    array_shift($csv);
}

/**
 * Validate CSV
 * @param array $data User/form data
 * @throws ValidationException
 */
function validate($data) {
    $errors = [];
    foreach ($data['csv'] as $i => $row) {
        if (empty($row['firstname']) && empty($row['lastname']) && empty($row['company'])) {
            $errors[] = 'Row #' . ($i + 1) . ": Either company, firstname or lastname must be set.";
        }
    }
    if (!empty($errors)) {
        throw new ValidationException($errors);
    }
}

/**
 * Post contacts
 * @param array $data User/form data
 * @throws Exception
 */
function post($data) {
    $errors = [];
    foreach ($data['csv'] as $i => $row) {
        $params = [];
        if (!empty($row['nr'])) {
            $params['nr'] = $row['nr'];
        }
        if (!empty($row['company'])) {
            $params['company'] = $row['company'];
        }
        if (!empty($row['title'])) {
            $params['titleId'] = getTitleId($data, $row['title']);
        }
        if (!empty($row['firstname'])) {
            $params['firstName'] = $row['firstname'];
        }
        if (!empty($row['lastname'])) {
            $params['lastName'] = $row['lastname'];
        }
        $contacts = [];
        if (!empty($row['email_work'])) {
            $contacts[] = ['type' => 'EMAIL', 'purpose' => 'WORK', 'address' => $row['email_work']];
        }
        if (!empty($row['email_private'])) {
            $contacts[] = ['type' => 'EMAIL', 'purpose' => 'PRIVATE', 'address' => $row['email_private']];
        }
        if (!empty($row['phone_work'])) {
            $contacts[] = ['type' => 'PHONE', 'purpose' => 'WORK', 'address' => $row['phone_work']];
        }
        if (!empty($row['phone_private'])) {
            $contacts[] = ['type' => 'PHONE', 'purpose' => 'PRIVATE', 'address' => $row['phone_private']];
        }
        if (!empty($contacts)) {
            $params['contacts'] = json_encode($contacts);
        }
        if (is_numeric(CATEGORY_ID)) {
            $params['categoryId'] = CATEGORY_ID;
        }
        $addresses = [];
        if (!empty($row['address']) || !empty($row['zip']) || !empty($row['city']) || !empty($row['country'])) {
            $address = ['type' => 'MAIN'];
            if (!empty($row['address'])) {
                $address['address'] = $row['address'];
            }
            if (!empty($row['zip'])) {
                $address['zip'] = ZIP_NUMERIC_ONLY ? preg_replace("/\D/", "", $row['zip']) : $row['zip'];
            }
            if (!empty($row['city'])) {
                $address['city'] = $row['city'];
            }
            if (!empty($row['country'])) {
                $address['country'] = strtoupper($row['country']);
            }
            $addresses[] = $address;
        }
        if (!empty($addresses)) {
            $params['addresses'] = json_encode($addresses);
        }
        $custom = getCustomFields($row);
        if (!empty($custom)) {
            $params['custom'] = $custom;
        }
        try {
            curlPost(BASE_URL . '/person/create.json', $params);
        } catch (ValidationException $e) {
            $errors[] = "Row #" . ($i + 1) . ": " . implode(", ", $e->errors);
        }
        // to avoid 429 Too Many Requests
        if ($i > 0 && $i % 7 == 0) {
            sleep(2);
        }
    }
    if (!empty($errors)) {
        throw new ValidationException($errors);
    }
}

/**
 * Get custom fields
 * @param array $row Row
 * @return string Custom fields XML
 */
function getCustomFields($row) {
    $hasCustomFields = false;
    $custom = "<values>";
    foreach ($row as $key => $value) {
        if (strpos($key, 'customField') === 0) {
            $hasCustomFields = true;
            $num = preg_replace("/\D/", "", $key);
            $custom .= "<customField" . $num . ">" . he($value) . "</customField" . $num . ">";
        }
    }
    $custom .= "</values>";
    return $hasCustomFields ? $custom : null;
}

/**
 * Get title ID from title
 * @param array $data User/form data
 * @param string $title Title
 * @return int|null Title ID
 * @throws Exception
 */
function getTitleId(&$data, $title) {
    $lcTitle = trim(strtolower($title));
    if (!isset($data['titleMap'])) {
        $data['titleMap'] = [];
        $list = curlGet(BASE_URL . '/person/title/list.json');
        foreach ($list['data'] as $row) {
            $names = getArrayFromI18n($row['name']);
            foreach ($names as $name) {
                $data['titleMap'][trim(strtolower($name))] = $row['id'];
            }
            $data['titleMap'][$row['id']] = $row['id'];
        }
    }
    return isset($data['titleMap'][$lcTitle]) ? $data['titleMap'][$lcTitle] : null;
}

/**
 * Get values array from i18n xml
 * @param string $xml XML string
 * @return array Values
 */
function getArrayFromI18n($xml) {
    if (strpos($xml, '<values>') === 0) {
        $parser = xml_parser_create();
        xml_parse_into_struct($parser, $xml, $values);
        xml_parser_free($parser);
        $arr = [];
        foreach ($values as $value) {
            if (isset($value['value'])) {
                $arr[] = $value['value'];
            }
        }
        return $arr;
    }
    return [$xml];
}

/**
 * Check if CLI (command-line interface) or not
 * @return bool CLI or not
 */
function isCli() {
    return PHP_SAPI == "cli";
}

/**
 * HTMLEntities wrapper
 * @param string $text Text
 * @return string Encoded text
 */
function he($text) {
    return htmlentities($text, null, 'utf-8');
}

/**
 * Throw exception for remote validation errors
 * @param array $response Response object
 * @param string $url URL
 * @throws ValidationException
 */
function throwRemoteValidation($response, $url = '') {
    if (!$response['success'] && isset($response['errors'])) {
        $errors = [];
        if (!empty($url)) {
            $errors[] = $url;
        }
        foreach ($response['errors'] as $error) {
            $errors[] = $error['field'] ? $error['field'] . ': ' . $error['message'] : $error['message'];
        }
        throw new ValidationException($errors);
    }
}

/**
 * Remove UTF-8 BOM from CSV file
 * @param string $text Text
 * @return string Cleaned text
 */
function removeUtf8Bom($text) {
    if (substr($text, 0, 3) == chr(hexdec('EF')) . chr(hexdec('BB')) . chr(hexdec('BF'))) {
        return substr($text, 3);
    } else {
        return $text;
    }
}

/**
 * cURL get request (and decode JSON response)
 * @param string $url URL
 * @param array $params Parameters to send
 * @return array List
 * @throws ValidationException
 * @throws Exception
 */
function curlGet($url, $params = []) {
    $result = curl($url, API_KEY, '', $params);
    $response = json_decode($result['response'], true);
    throwRemoteValidation($response, $url);
    return $response;
}

/**
 * cURL post request (and decode JSON response)
 * @param string $url URL
 * @param array $params Parameters to send
 * @return array Response
 * @throws ValidationException
 * @throws Exception
 */
function curlPost($url, $params = []) {
    $result = curl($url, API_KEY, '', $params, 'POST');
    $response = json_decode($result['response'], true);
    throwRemoteValidation($response, $url);
    return $response;
}

/**
 * cURL wrapper
 * @param string $url URL
 * @param string $user Basic Auth Username
 * @param string $password Basic Auth Password
 * @param array $params Parameters to send
 * @param string $method HTTP Method
 * @return array Status and response
 * @throws Exception
 */
function curl($url, $user = '', $password = '', $params = [], $method = 'GET') {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_USERPWD, $user . ':' . $password);
    if ($method == 'POST') {
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($params));
    } else {
        curl_setopt($ch, CURLOPT_URL, !empty($params) ? $url . '?' . http_build_query($params) : $url);
    }
    // uncomment this line if you have a cacert.pem file to verify SSL host/peer
    // curl_setopt($ch, CURLOPT_CAINFO, realpath(dirname(__FILE__)) . DIRECTORY_SEPARATOR . 'cacert.pem');
    //curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    //curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $response = curl_exec($ch);
    if ($response === false) {
        throw new Exception(curl_error($ch));
    }
    $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    if ($status == 401) {
        throw new Exception("401 - No valid API key provided.");
    }
    if ($status == 403) {
        throw new Exception("403 - The API key doesn't have permissions to perform the request.");
    }
    if ($status == 404) {
        throw new Exception("404 - The requested endpoint doesn't exist.");
    }
    if ($status == 429) {
        throw new Exception("429 - Too many requests hit the API too quickly.");
    }
    if ($status >= 500) {
        throw new Exception($status . " - Internal server error.");
    }
    curl_close($ch);
    return array('status' => $status, 'response' => $response);
}

/**
 * Validation exception
 */
class ValidationException extends Exception {
    public $errors = [];

    /**
     * Constructor
     * @param array $errors Error messages
     */
    public function __construct($errors = []) {
        parent::__construct();
        $this->errors = $errors;
    }
}