<?php
// configuration
const API_KEY = 'zjdqcZjsfSYyM4EPG19tHss6X5NIHf0b';
const ORGANIZATION = 'myorg';
const BASE_URL = 'https://' . ORGANIZATION . '.cashctrl.com/api/v1';

$data = [
    'company' => '',
    'titleId' => '2',
    'firstname' => '',
    'lastname' => '',
    'address' => '',
    'zip' => '',
    'city' => '',
    'email' => '',
    'phone' => '',
    'message' => ''
];
$result = ['success' => '', 'errors' => []];

// get user input, validate and post data
if (input($data)) {
    try {
        validate($data);
        post($data);
        $result['success'] = 'Vielen Dank für Ihre Anfrage! Wir werden uns sobald wie möglich bei Ihnen melden.';
    } catch (Exception $e) {
        if ($e instanceof ValidationException) {
            $result['errors'] = $e->errors;
        }
        if (!empty($e->getMessage())) {
            $result['errors'][] = $e->getMessage();
        }
    }
}

// show result in CLI
if (isCli()) {
    if ($result['success']) {
        print $result['success'] . "\n";
    }
    if (!empty($result['errors'])) {
        print implode("\n", $result['errors']) . "\n";
    }
    die();
}

// show form and result in WEB
header("Content-type: text/html; charset=utf-8");
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <title>Offertanfrage</title>
    <style>
        * {box-sizing:border-box}
        body {color:#444;background:white;padding:40px 20px}
        body, input, textarea {font:16px/1.5 Helvetica, Arial, sans-serif;}
        main {max-width:600px;margin:0 auto}
        form .row {margin-bottom:15px}
        form .row.columns {display:flex;justify-content:space-between;align-items:stretch}
        form .row.columns label:first-child {padding-right:5px}
        form .row.columns label:last-child {padding-left:5px}
        form .row.col-50 label {flex:0 0 50%;max-width:50%}
        form .row.col-33-66 label:first-child {flex:0 0 33.3333%;max-width:33.3333%}
        form .row.col-33-66 label:last-child {flex:0 0 66.6666%;max-width:66.6666%}
        input, textarea {padding:8px}
        input[type=text], textarea {border:1px solid #CCC;background:white;width:100%}
        .success, .errors {margin-bottom:20px}
        .success {color:green;font-size:21px}
        .errors {color:red}
    </style>
</head>
<body>

<main>
    <h1>Offertanfrage</h1>
    <form method="post" action="?">
        <div class="success"><?= he($result['success']) ?></div>
        <div class="errors"><?= implode('<br>', $result['errors']) ?></div>
        <div class="row">
            <label><input type="text" name="company" value="<?= he($data['company']) ?>" placeholder="Firma"/></label>
        </div>
        <div class="row">
            <label><input type="radio" name="titleId" value="2" <?= $data['titleId'] == '2' ? 'checked' : '' ?>/> Frau</label> &nbsp;&nbsp;&nbsp;
            <label><input type="radio" name="titleId" value="1" <?= $data['titleId'] == '1' ? 'checked' : '' ?>/> Herr</label>
        </div>
        <div class="row columns col-50">
            <label><input type="text" name="firstname" value="<?= he($data['firstname']) ?>" placeholder="Vorname *"/></label>
            <label><input type="text" name="lastname" value="<?= he($data['lastname']) ?>" placeholder="Nachname *"/></label>
        </div>
        <div class="row">
            <label><input type="text" name="address" value="<?= he($data['address']) ?>" placeholder="Strasse / Nr. *"/></label>
        </div>
        <div class="row columns col-33-66">
            <label><input type="text" name="zip" value="<?= he($data['zip']) ?>" placeholder="PLZ *"/></label>
            <label><input type="text" name="city" value="<?= he($data['city']) ?>" placeholder="Ort *"/></label>
        </div>
        <div class="row">
            <label><input type="text" name="email" value="<?= he($data['email']) ?>" placeholder="E-Mail *"/></label>
        </div>
        <div class="row">
            <label><input type="text" name="phone" value="<?= he($data['phone']) ?>" placeholder="Telefon"/></label>
        </div>
        <div class="row">
            <label><textarea name="message" placeholder="Mitteilung *" rows=7><?= he($data['message']) ?></textarea></label>
        </div>
        <div class="row">
            <input type="submit" value="Abschicken"/>
        </div>
    </form>
</main>

</body>
</html>
<?php

// FUNCTIONS

/**
 * Get input from user (either input via CLI or from WEB form)
 * @param array $data User/form data
 * @return bool User has entered data or not
 */
function input(&$data) {
    if (isCli()) {
        foreach (array_keys($data) as $key) {
			if (empty($data[$key])) {
				$value = readline($key . ": ");
				$data[$key] = $value;
			}
        }
        return true;
    }
    if ($_POST) {
        foreach (array_keys($data) as $key) {
            $data[$key] = isset($_POST[$key]) ? $_POST[$key] : '';
        }
        return true;
    }
    return false;
}

/**
 * Validate offer request
 * @param array $data User/form data
 * @throws ValidationException
 */
function validate($data) {
    $errors = [];
    if (!in_array($data['titleId'], ['1', '2'])) {
        $errors[] = 'Anrede fehlt.';
    }
    if (empty($data['firstname'])) {
        $errors[] = 'Vorname fehlt.';
    }
    if (empty($data['lastname'])) {
        $errors[] = 'Nachname fehlt.';
    }
    if (empty($data['address'])) {
        $errors[] = 'Strasse/Nr. fehlt.';
    }
    if (empty($data['zip'])) {
        $errors[] = 'PLZ fehlt.';
    } else if (!is_numeric($data['zip'])) {
        $errors[] = 'PLZ muss numerisch sein.';
    }
    if (empty($data['city'])) {
        $errors[] = 'Ort fehlt.';
    }
    if (empty($data['email'])) {
        $errors[] = 'E-Mail Adresse fehlt.';
    } else if (!filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
        $errors[] = 'E-Mail Adresse ist ungültig.';
    }
    if (empty($data['message'])) {
        $errors[] = 'Mitteilung fehlt.';
    }
    if (!empty($errors)) {
        throw new ValidationException($errors);
    }
}

/**
 * Post offer request
 * @param array $data User/form data
 * @throws Exception
 */
function post($data) {
    $response = curlPost(BASE_URL . '/order/create.json', [
        'associateId' => getOrCreatePerson($data),
        'categoryId' => getOrCreateOrderCategory(),
        'date' => date('Y-m-d'),
        'sequenceNumberId' => getOrCreateSequenceNumber(),
        'description' => 'Anfrage über www.myorg.com'
    ]);
    $orderId = $response['insertId'];
    curlPost(BASE_URL . '/order/document/update.json', [
        'id' => $orderId,
        'header' => nl2br($data['message'])
    ]);
}

/**
 * Get or create person (customer who requested an offer)
 * @param array $data User/form data
 * @return int|null Person ID
 * @throws Exception
 */
function getOrCreatePerson($data) {
    $personId = null;
    // identify person by email address
    $list = curlGet(BASE_URL . '/person/list.json?query=' . $data['email']);
    if ($list['total'] > 0) {
        $personId = $list['data'][0]['id'];
    }
    // create person
    if (!$personId) {
        $contacts = [['type' => 'EMAIL', 'purpose' => 'WORK', 'address' => $data['email']]];
        if (!empty($data['phone'])) {
            $contacts[] = ['type' => 'PHONE', 'purpose' => 'WORK', 'address' => $data['phone']];
        }
        $response = curlPost(BASE_URL . '/person/create.json', [
            'company' => $data['company'],
            'titleId' => $data['titleId'],
            'firstName' => $data['firstname'],
            'lastName' => $data['lastname'],
            'addresses' => json_encode([
                ['type' => 'MAIN', 'address' => $data['address'], 'zip' => $data['zip'], 'city' => $data['city']]
            ]),
            'contacts' => json_encode($contacts)
        ]);
        $personId = $response['insertId'];
    }
    return $personId;
}

/**
 * Get or create order category 'Offertanfrage'
 * @throws Exception
 */
function getOrCreateOrderCategory() {
    $categoryId = null;
    $firstCategoryId = null;
    $list = curlGet(BASE_URL . '/order/category/list.json?type=SALES');
    foreach ($list['data'] as $category) {
        if ($firstCategoryId == null) {
            $firstCategoryId = $category['id'];
        }
        if (strpos($category['nameSingular'], 'Offer request') !== false) {
            $categoryId = $category['id'];
            break;
        }
    }
    // create order category
    if (!$categoryId) {
        $response = curlPost(BASE_URL . '/order/category/create.json', [
            'accountId' => 4,
            'nameSingular' => '<values><de>Offertanfrage</de><en>Offer request</en></values>',
            'namePlural' => '<values><de>Offertanfragen</de><en>Offer requests</en></values>',
            'status' => json_encode([
                ['icon' => 'BLUE', 'name' => '<values><de>Offen</de><en>Open</en></values>'],
                ['icon' => 'GREEN', 'name' => '<values><de>Offeriert</de><en>Offered</en></values>'],
                ['icon' => 'YELLOW', 'name' => '<values><de>Ignoriert</de><en>Ignored</en></values>']
            ]),
            'dueDays' => 7,
            'hasDueDays' => true,
            'isDisplayPrices' => true,
            'isSwitchRecipient' => true,
            'sequenceNrId' => getOrCreateSequenceNumber()
        ]);
        $categoryId = $response['insertId'];
        curlPost(BASE_URL . '/order/category/reorder.json', ['ids' => $categoryId, 'target' => $firstCategoryId]);
    }
    return $categoryId;
}

/**
 * Get or create sequence number for 'Offertanfragen'
 * @return int|null Sequence number ID
 * @throws Exception
 */
function getOrCreateSequenceNumber() {
    $sequenceNumberId = null;
    $list = curlGet(BASE_URL . '/sequencenumber/list.json');
    foreach ($list['data'] as $sequenceNumber) {
        if (strpos($sequenceNumber['name'], 'Offertanfragen') !== false) {
            $sequenceNumberId = $sequenceNumber['id'];
            break;
        }
    }
    // create sequence number
    if (!$sequenceNumberId) {
        $response = curlPost(BASE_URL . '/sequencenumber/create.json', [
            'name' => 'Offertanfragen',
            'pattern' => 'OA-$y$m$d$nd',
            'yearDigits' => 2
        ]);
        $sequenceNumberId = $response['insertId'];
    }
    return $sequenceNumberId;
}

/**
 * Check if CLI (command-line interface) or not
 * @return bool CLI or not
 */
function isCli() {
    return PHP_SAPI == "cli";
}

/**
 * HTMLEntities wrapper
 * @param string $text Text
 * @return string Encoded text
 */
function he($text) {
    return htmlentities($text, null, 'utf-8');
}

/**
 * Throw exception for remote validation errors
 * @param array $response Response object
 * @param string $url URL
 * @throws Exception
 */
function throwRemoteValidation($response, $url = '') {
    if (!$response['success'] && isset($response['errors'])) {
        $errors = [];
        if (!empty($url)) {
            $errors[] = $url;
        }
        foreach ($response['errors'] as $error) {
            $errors[] = $error['field'] ? $error['field'] . ': ' . $error['message'] : $error['message'];
        }
        throw new ValidationException($errors);
    }
}

/**
 * cURL get request (and decode JSON response)
 * @param string $url URL
 * @param array $params Parameters to send
 * @return array List
 * @throws Exception
 */
function curlGet($url, $params = []) {
    $result = curl($url, API_KEY, '', $params);
    $response = json_decode($result['response'], true);
    throwRemoteValidation($response, $url);
    return $response;
}

/**
 * cURL post request (and decode JSON response)
 * @param string $url URL
 * @param array $params Parameters to send
 * @return array Response
 * @throws Exception
 */
function curlPost($url, $params = []) {
    $result = curl($url, API_KEY, '', $params, 'POST');
    $response = json_decode($result['response'], true);
    throwRemoteValidation($response, $url);
    return $response;
}

/**
 * cURL wrapper
 * @param string $url URL
 * @param string $user Basic Auth Username
 * @param string $password Basic Auth Password
 * @param array $params Parameters to send
 * @param string $method HTTP Method
 * @return array Status and response
 * @throws Exception
 */
function curl($url, $user = '', $password = '', $params = [], $method = 'GET') {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_USERPWD, $user . ':' . $password);
    if ($method == 'POST') {
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($params));
    } else {
        curl_setopt($ch, CURLOPT_URL, !empty($params) ? $url . '?' . http_build_query($params) : $url);
    }
    // uncomment this line if you have a cacert.pem file to verify SSL host/peer
    // curl_setopt($ch, CURLOPT_CAINFO, realpath(dirname(__FILE__)) . DIRECTORY_SEPARATOR . 'cacert.pem');
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $response = curl_exec($ch);
    if ($response === false) {
        throw new Exception(curl_error($ch));
    }
    $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    if ($status == 401) {
        throw new Exception("401 - No valid API key provided.");
    }
    if ($status == 403) {
        throw new Exception("403 - The API key doesn't have permissions to perform the request.");
    }
    if ($status == 404) {
        throw new Exception("404 - The requested endpoint doesn't exist.");
    }
    if ($status == 429) {
        throw new Exception("429 - Too many requests hit the API too quickly.");
    }
    if ($status >= 500) {
        throw new Exception($status . " - Internal server error.");
    }
    curl_close($ch);
    return array('status' => $status, 'response' => $response);
}

/**
 * Validation exception
 */
class ValidationException extends Exception {
    public $errors = [];

    /**
     * Constructor
     * @param array $errors Error messages
     */
    public function __construct($errors = []) {
        parent::__construct();
        $this->errors = $errors;
    }
}